/*
 * FireDoor.h
 *
 * Simple firewall proxy service for internet access under Windows NT.
 *
 * Copyright 1995-1996 Equivalence
 */

#include <svcproc.h>
#include <sockets.h>
#include <remconn.h>
#include <http.h>
#include <cypher.h>


///////////////////////////////////////////////////////////////////////////////

PDECLARE_CLASS(SocketSpec, PObject)
  public:
    SocketSpec() {}
    SocketSpec(PIPSocket & ip)
        { ip.GetPeerAddress(address); portNumber = ip.GetPort(); }
    SocketSpec(PIPSocket::Address addr, WORD port = 0)
        : address(addr), portNumber(port) { }

    virtual PObject * Clone() const;
    virtual Comparison Compare(const PObject & obj) const;
    virtual PINDEX HashFunction() const;

    BOOL SetSpec(const PString & spec, WORD defPort);
    BOOL WildcardMatch(const SocketSpec & spec) const;

    const PIPSocket::Address & GetHost() const { return address; }
    WORD GetPort() const { return portNumber; }

    operator PString() const;

  protected:
    PIPSocket::Address address;
    WORD               portNumber;
};

PSORTED_LIST(SocketSpecList, SocketSpec);


PDICTIONARY(SocketSpecDict, PString, SocketSpec);


PDECLARE_CLASS(Validation, SocketSpec)
  public:
    SocketSpecList allow;
};

PSORTED_LIST(ValidationList, Validation);


///////////////////////////////////////////////////////////////////////////////

PDECLARE_CLASS(SocketSpecField, PHTTPStringField)
  public:
    SocketSpecField(
      const char * name
    );

    virtual BOOL Validated(
      const PString & newVal,
      PStringStream & msg
    ) const;
};


///////////////////////////////////////////////////////////////////////////////

PDECLARE_CLASS(ServiceField, PHTTPStringField)
  public:
    ServiceField(
      const char * name
    );

    virtual BOOL Validated(
      const PString & newVal,
      PStringStream & msg
    ) const;
};


///////////////////////////////////////////////////////////////////////////////

PDECLARE_CLASS(UserInfo, PObject)
  public:
    UserInfo() { dataSent = dataReceived = 0; }

    DWORD GetDataSent() const { return dataSent; }
    DWORD GetDataReceived() const { return dataReceived; }

    void AddDataSent(DWORD amt) { dataSent += amt; }
    void AddDataReceived(DWORD amt) { dataReceived += amt; }

    PTimer timer;

  private:
    DWORD dataSent;
    DWORD dataReceived;
};

PDICTIONARY(UserDict, SocketSpec, UserInfo);


///////////////////////////////////////////////////////////////////////////////

class ConfigPage;

PDECLARE_CLASS(RemoteSystem, PObject)
  public:
    RemoteSystem(PConfig & cfg, ConfigPage * rsrc);

    const PString & GetRemoteName() const { return remoteName; }
    const PTimeInterval & GetDisconnectTime() const { return disconnectTime; }
    const PTimeInterval & GetConnectLimit() const { return connectTimeLimit; }
    const PTimeInterval & GetConnectPeriod() const { return connectTimePeriod; }
    const PTimeInterval & GetConnectTime() const { return totalConnectTime; }
    PINDEX GetDialRetries() const { return dialRetries; }

    BOOL GetRelayOverride(const PString & desc, SocketSpec & sock) const;

    void AccumulateConnectTime(const PTime & connectTime)
      { totalConnectTime += (PTime() - connectTime); }

  private:
    PString        remoteName;
    PTimeInterval  disconnectTime;
    PTimeInterval  connectTimeLimit;
    PTimeInterval  connectTimePeriod;
    PTimeInterval  totalConnectTime;
    PINDEX         dialRetries;
    SocketSpecDict relayOverrides;
};

PLIST(RemoteSystemList, RemoteSystem);


///////////////////////////////////////////////////////////////////////////////

class ProxyThread;
PLIST(ThreadList, ProxyThread);

PLIST(RealSocketList, PIPSocket);

PDECLARE_CLASS(SocketList, RealSocketList)
  public:
    void Append(PIPSocket * socket);
    void Remove(PIPSocket * socket);
    void Shutdown();
  private:
    PSemaphore mutex;
};


PDECLARE_CLASS(Proxy, PServiceProcess)
  public:
    Proxy();

    virtual void Main();

    typedef void (Proxy::*Handler)(ProxyThread &, PIPSocket &);

    void HandleUDP(ProxyThread & thread, PIPSocket & listener);
    void HandleTCP(ProxyThread & thread, PIPSocket & listener);
    void HandleSOCKS(ProxyThread & thread, PIPSocket & listener);
    void HandleHTTP(ProxyThread & thread, PIPSocket & listener);
    void HandleFTP(ProxyThread & thread, PIPSocket & listener);

    virtual BOOL OnStart();
    virtual void OnStop();
    virtual BOOL OnPause();
    virtual void OnContinue();
    BOOL Initialise(const char * initMsg);

  private:
    BOOL StartListener(const PString & service, const PString & relay);
    BOOL Validated(const SocketSpec & inside, const SocketSpec & outside);
    void LinkSockets(PIPSocket & in, PIPSocket & out, const PString & msg);
    void DemandDial(ProxyThread & thread);
    BOOL DialRemoteSystem();


    ThreadList waitingThreads;
    PSemaphore waitingMutex;

    SocketList listeners;
    SocketList activeSockets;

    UserDict   users;
    PINDEX     maxUsers;
    PSemaphore usersMutex;

    SocketSpecDict relays;
    ValidationList proxyValidation;

    PTimeInterval  dgramTimeout;
    PINDEX         dgramRetries;
    PTimeInterval  linkTimeout;
    PTimeInterval  socksTimeout;

    PHTTPSpace      httpSpace;

    RemoteSystemList  remoteSystems;
    PINDEX            remoteSelected;
    PRemoteConnection remoteConnection;
    PTime             remoteConnectTime;
    PTimer            remoteTimer;
    PSemaphore        remoteDial;
    PSemaphore        remoteMutex;

    enum {
      SOCKS_CONNECT = 1,
      SOCKS_BIND = 2,

      SOCKS_RESULT = 90,
      SOCKS_FAIL = 91,
      SOCKS_NO_IDENTD = 92, // Failed to connect to Identd on client machine
      SOCKS_BAD_ID = 93,    // Client's Identd reported a different user-id

      SOCKS_VERSION = 4
    };

  friend class ControlPage;
  friend class ProxyThread;
};


///////////////////////////////////////////////////////////////////////////////

PDECLARE_CLASS(ProxyThread, PThread)
  public:
    ProxyThread(Proxy & proc, PIPSocket & listen, Proxy::Handler handle);
    PSemaphore remoteDial;
  private:
    void Main();
    Proxy &        process;
    PIPSocket &    listener;
    Proxy::Handler handler;
};


///////////////////////////////////////////////////////////////////////////////

PDECLARE_CLASS(ConfigPage, PHTTPConfig)
  public:
    ConfigPage(
      Proxy & prox,
      const PString & section,
      const PHTTPAuthority & auth
    );

    virtual void OnPOST(
      PHTTPSocket & socket,         // HTTP socket that received the request
      const PURL & url,             // Universal Resource Locator for document.
      const PMIMEInfo & info,       // Extra MIME information in command.
      const PStringToString & data  // Variables in the POST data.
    );
    virtual BOOL Post(
      PHTTPRequest & request,       // Information on this request.
      const PStringToString & data, // Variables in the POST data.
      PHTML & replyMessage          // Reply message for post.
    );

  protected:
    ConfigPage(
      Proxy & prox,
      const PString & title,
      const PString & section,
      const PHTTPAuthority & auth
    );

    virtual BOOL GetExpirationDate(
      PTime & when          // Time that the resource expires
    );

  private:
    Proxy & proxy;
    BOOL reload;
};


///////////////////////////////////////////////////////////////////////////////

PDECLARE_CLASS(RemoteListPage, ConfigPage)
  public:
    RemoteListPage(
      Proxy & prox,
      const PHTML & str,
      const PHTTPAuthority & auth
    );

    virtual BOOL Post(
      PHTTPRequest & request,       // Information on this request.
      const PStringToString & data, // Variables in the POST data.
      PHTML & replyMessage          // Reply message for post.
    );
};


///////////////////////////////////////////////////////////////////////////////

PDECLARE_CLASS(RegisterPage, ConfigPage)
  public:
    RegisterPage(
      Proxy & prox,
      const PSecureConfig & sconf,
      const PHTTPAuthority & auth
    );

    virtual BOOL Post(
      PHTTPRequest & request,       // Information on this request.
      const PStringToString & data, // Variables in the POST data.
      PHTML & replyMessage          // Reply message for post.
    );

  private:
    PStringArray securedKeys;
};


///////////////////////////////////////////////////////////////////////////////

PDECLARE_CLASS(ControlPage, PHTTPString)
  public:
    ControlPage(Proxy & prox, PHTTPAuthority & auth);

    virtual void OnLoadedText(
      PHTTPRequest & request,    // Information on this request.
      PString & text             // Data used in reply.
    );

    virtual BOOL Post(
      PHTTPRequest & request,       // Information on this request.
      const PStringToString & data, // Variables in the POST data.
      PHTML & replyMessage          // Reply message for post.
    );

  private:
    Proxy & proxy;
};


// End of File ///////////////////////////////////////////////////////////////
