// This is the NT version of Xroach.  3/94
//
// Is was ported and adapted to Windows NT by Mike Berg, mberg@netcom.com
//
// Some Information on the changes that were made is contained is
// the README file.
//
// See below for Copyright information from the original author:
//
// Because "Hungarian Notation" is used exclusively in Microsoft
// documentation of the Win32 API, it was easier to allow
// some of it to creep into this program than to banish it
// entirely.  An attempt was made to keep it to a minimum.
/*
    Xroach - A game of skill.  Try to find the roaches under your windows.
    
    Copyright 1991 by J.T. Anderson

    jta@locus.com
    
    This program may be freely distributed provided that all
    copyright notices are retained.

    To build:
      cc -o xroach roach.c -lX11 [-lsocketorwhatever] [-lm] [-l...]

    Dedicated to Greg McFarlane.   (gregm@otc.otca.oz.au)
    
    Squish option contributed by Rick Petkiewizc (rick@locus.com)
    
    Virtual root code adapted from patch sent by Colin Rafferty who
    borrowed it from Tom LaStrange.  Several other folks sent similar
    fixes.
*/

/* @(#)xroach.c	1.5 4/2/91 11:53:31 */

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "resource.h"
#include <math.h>
#include <signal.h>
char Copyright[] = "Xroach\nCopyright 1991 J.T. Anderson";

#include "NTRoach.h" /* roachmap.h was shortened for FAT filesystem */

#define NTRoachVersion "NTRoach v. 1.1" /* displayed on Settings dialog */

/* CBTHook library and library routines */
#include "events.h"
#define NTRHook "NTRHook.DLL"
HANDLE hNTRHook;
int (*InstallHook)();
int (*UnInstallHook)();

/* Registry stuff */
HKEY hRoachKey;
char *szRoachKey = "Software\\Freeware\\NTRoach";
char *szNumRoaches = "NumRoaches";
char *szSpeedRoaches = "SpeedRoaches";
char *szRoachColor = "RoachColor";
char *szSquishColor = "RoachGutsColor";


typedef struct Roach {
    RoachMap *rp;
    int index; 		/* index into the RoachMap */
    float x;
    float y;
    int intX;
    int intY;
    int hidden;
    int turnLeft;
    int steps;
} Roach;

Roach *roaches;
int maxRoaches = 10;
int curRoaches = 0;
float roachSpeed = 20.0F;

unsigned int display_width, display_height;
int center_x, center_y;

int newMaxRoaches;
int newSpeed;

#define ARM_ROACH_SQUISHER	0x9020
#define ROACH_CONFIG		0x9030
#define HIDDEN  		0x9040
#define NOT_HIDDEN  		0x9050

HWND hDesktopWnd;
HDC hDesktopDC;
HDC hRoachDC;
RECT rDesktopRect;
HRGN hRootVisibleRgn;

/* ThreadId of the main loop */
DWORD dwThreadId; 

HANDLE hNeedCalcEvent;
HANDLE hCheckSquishEvent;
HANDLE hNewNumRoachesEvent;

HWND hMainWnd;
BOOL bMainWnd;
HMENU hSystemMenu;

HICON hRoachIcon;
HICON hSquishIcon;

int squishBits = IDB_SQUISH; /* Identifier of the BMP resource */
HBITMAP hSquishBitMap;
HDC hSquishDC;

POINT pMouse;

BOOL	bNewRoachColor;
COLORREF crNewRoachColor;
COLORREF crRoachColor = RGB(0,0,0); /* default Roach color */

BOOL	bNewSquishColor;
COLORREF crNewSquishColor;
COLORREF crSquishColor  = RGB(0,255,0); /* default Squish color */

HBRUSH  hRoachBrush;
HBRUSH  hSquishBrush;

void
SigHandler(int i)
{

#ifdef CONSOLE_APP
    printf("NTRoach exiting due to signal %d...\n",i);
#endif
    UnInstallHook();
    exit(0);
}

void
printErrorInfo(char *programMsg,DWORD dwErrorId)
{
    char errorMsgBuf[512];
    char outputBuf[1024];

    FormatMessage(
	FORMAT_MESSAGE_FROM_SYSTEM,
	NULL, dwErrorId,
	MAKELANGID(0, SUBLANG_ENGLISH_US),
	errorMsgBuf, sizeof(errorMsgBuf), NULL);

    sprintf(outputBuf,"%s -- Error %d: %s",
    			programMsg, dwErrorId, errorMsgBuf);
#ifdef CONSOLE_APP
    printf(outputBuf);
#else
    MessageBox(bMainWnd?hMainWnd:NULL  ,outputBuf,"NTRoach Error",MB_OK);
#endif
}

/*
   Generate random integer between 0 and maxVal-1.
*/
int
RandInt(maxVal)
int maxVal;
{
	return rand() % maxVal;
}

/*
   Check for roach completely in specified rectangle.
*/
int
RoachInRect(roach, rx, ry, x, y, width, height)
Roach *roach;
int rx;
int ry;
int x;
int y;
unsigned int width;
unsigned int height;
{
    if (rx < x) return 0;
    if ((rx + roach->rp->width) > (x + width)) return 0;
    if (ry < y) return 0;
    if ((ry + roach->rp->height) > (y + height)) return 0;
    
    return 1;
}




BOOL CALLBACK
EnumWindowsProc(HWND hWnd, LPARAM lParam) 
{
    BOOL bResult;
    RECT rRect;
    DWORD dwStyle;
    HRGN hTmpRgn;
    int iResult;
    char szClassName[80];
    bResult = GetClassName(hWnd,szClassName,sizeof(szClassName));
    if (!bResult) {
      printErrorInfo("GetClassName",GetLastError());
      return TRUE;
    }
    if (strcmp(szClassName,"ConsoleWindowClass") == 0) {
	return TRUE;
    }
    else {
	dwStyle = GetWindowLong(hWnd,GWL_STYLE);
	if (dwStyle & WS_VISIBLE) {
	    bResult = GetWindowRect(hWnd,&rRect);
	    if(!bResult) {
		printErrorInfo("GetWindowRect",GetLastError());
		return TRUE;
	    }
	    hTmpRgn = CreateRectRgn(
	    	rRect.left,rRect.top,rRect.right,rRect.bottom);
	    if (!hTmpRgn) {
		printErrorInfo("CreateRectRgn",GetLastError());
		return TRUE;
	    }
	    iResult = CombineRgn(hRootVisibleRgn, hRootVisibleRgn, hTmpRgn, 
			RGN_DIFF );
	    if (iResult == ERROR) {
		 printErrorInfo("CombineRgn",GetLastError());
		 DeleteObject(hTmpRgn);
		 return TRUE;
	    }
	    DeleteObject(hTmpRgn);
	}
    }
    return TRUE;
}


/*
   Calculate Visible region of root window.
*/
int
CalcRootVisible()
{
    int wx;
    BOOL bResult;

    /* MJB bug alert */
    /* Sometimes we've been told to recalculate visible region and 
     * Win32 gives us stale data. This has occured when it was
     * our very own thread that was responsible for the recalculation
     * request. Sleeping for a little while seems to fix things up.
     */ 
    Sleep(10);
    ResetEvent(hNeedCalcEvent);
    if (hRootVisibleRgn) {
	DeleteObject(hRootVisibleRgn);
    }
    hRootVisibleRgn = CreateRectRgn(
			rDesktopRect.left,rDesktopRect.top,
			rDesktopRect.right,rDesktopRect.bottom);
    if (!hRootVisibleRgn) {
	printErrorInfo("CreateRectRgn",GetLastError());
	exit(0);
    }


    bResult = EnumWindows(EnumWindowsProc,0);
    if (!bResult) {
	printErrorInfo("EnumWindows",GetLastError());
	exit(0);
    }
    
    /*
       Mark all roaches visible.
    */
    for (wx=0; wx<curRoaches; wx++)  {
			roaches[wx].hidden = 0;
    }
    return 0;
}

/*
 *   Check to see if we have to squish any roaches.
 */

void
CheckSquish()
{
    int x,y;
    int i;
    int rx;
    Roach *r;
    HBRUSH hOldBrush;

    ResetEvent(hCheckSquishEvent);
    x = pMouse.x;
    y = pMouse.y;
    for (rx=0; rx<curRoaches; rx++) {
	r = &roaches[rx];
	if (r->rp == NULL) continue;

	if (x > r->intX &&
		x < (r->intX + r->rp->width) &&
		y > r->intY &&
		y < (r->intY + r->rp->height)) {
	    hDesktopDC = GetDC(hDesktopWnd);
	    hOldBrush = SelectObject(hDesktopDC,hSquishBrush);
	    BitBlt(hDesktopDC, r->intX  , r->intY ,
			r->rp->width,
			r->rp->height,
			hSquishDC,
			0,
			0,
			0xE20746L );
	    SelectObject(hDesktopDC,hOldBrush);
	    ReleaseDC(hDesktopWnd, hDesktopDC);

	    /*
	     * Delete the roach
	     */

	    for (i = rx; i < curRoaches - 1; i++) {
		roaches[i] = roaches[i + 1];
	    }
	    curRoaches--;
	    rx--;
	}
    }
    if (!curRoaches) {
	MessageBox(hMainWnd,"No More Roaches!!","NTRoach",MB_OK);
    }
    return;
}


/*
   Mark hidden roaches.
*/
int
MarkHiddenRoaches()
{
    int rx;
    Roach *r;
    int nVisible;
    RECT rRect; 

    nVisible = 0;
    for (rx=0; rx<curRoaches; rx++) {
	r = &roaches[rx];
	
	if (!r->hidden) {
	    rRect.left = r->intX;
	    rRect.top = r->intY;
	    rRect.right =r->intX + r->rp->width;
	    rRect.bottom = r->intY + r->rp->height;
	    if (r->intX > 0 && 
		    !RectInRegion( hRootVisibleRgn, &rRect)) {
		r->hidden = 1;
	    }
	    else {
		nVisible++;
	    }
	}
    }
    return nVisible;
}


/*
   Check for roach overlapping specified rectangle.
*/
int
RoachOverRect(roach, rx, ry, x, y, width, height)
Roach *roach;
int rx;
int ry;
int x;
int y;
unsigned int width;
unsigned int height;
{
    if (rx >= (x + width)) return 0;
    if ((rx + roach->rp->width) <= x) return 0;
    if (ry >= (y + height)) return 0;
    if ((ry + roach->rp->height) <= y) return 0;
    
    return 1;
}



/*
   Give birth to a roach.
*/
void
AddRoach()
{
    Roach *r;
    
    if (curRoaches < maxRoaches) {
	r = &roaches[curRoaches++];
	r->index = RandInt(ROACH_HEADINGS);
	r->rp = &roachPix[r->index];
	r->x = RandInt(display_width - r->rp->width);
	r->y = RandInt(display_height - r->rp->height);
	r->intX = -1;
	r->intY = -1;
	r->hidden = 0;
	r->steps = RandInt(200);
	r->turnLeft = RandInt(100) >= 50;
    }
}

/*
	Alter the number of roaches.  Give birth or squish as necessary
*/
void
NewNumRoaches()
{
    int oldMaxRoaches;
    int i;

    ResetEvent(hNewNumRoachesEvent);
    oldMaxRoaches = maxRoaches;

    if (newMaxRoaches > maxRoaches) {
	maxRoaches = newMaxRoaches;
	roaches = (Roach *)realloc(roaches,sizeof(Roach) * maxRoaches);
	for(i=0;i<(newMaxRoaches - oldMaxRoaches);i++) {
	    AddRoach();
	}
    } 
    else if (newMaxRoaches < maxRoaches) {
	if (newMaxRoaches < curRoaches) {
	    int roachesToKill;
	    HBRUSH hOldBrush;
	    roachesToKill = curRoaches - newMaxRoaches;
	    hDesktopDC = GetDC(hDesktopWnd);
	    hOldBrush = SelectObject(hDesktopDC,hSquishBrush);
	    for (i=newMaxRoaches;i<curRoaches;i++){
		Roach *r;
		r = &roaches[i];
		BitBlt(hDesktopDC, r->intX  , r->intY ,
			    r->rp->width,
			    r->rp->height,
			    hSquishDC,
			    0,
			    0,
			    0xE20746L );

	    }
	    SelectObject(hDesktopDC,hOldBrush);
	    ReleaseDC(hDesktopWnd, hDesktopDC);
	    curRoaches = newMaxRoaches;
	}
	maxRoaches = newMaxRoaches;
	roaches = (Roach *)realloc(roaches,sizeof(Roach) * maxRoaches);
    }
}


/*
   Turn a roach.
*/
void
TurnRoach(roach)
Roach *roach;
{
    if (roach->index != (roach->rp - roachPix)) return;

    if (roach->turnLeft) {
	roach->index += (RandInt(30) / 10) + 1;
	if (roach->index >= ROACH_HEADINGS)
		roach->index -= ROACH_HEADINGS;
    }
    else {
	roach->index -= (RandInt(30) / 10) + 1;
	if (roach->index < 0)
	    roach->index += ROACH_HEADINGS;
    }
}

/*
   Move a roach.
*/
void
MoveRoach(rx)
int rx;
{
    Roach *roach;
    Roach *r2;
    float newX;
    float newY;
    int ii;
    
    roach = &roaches[rx];
    newX = roach->x + (roachSpeed * roach->rp->cosine);
    newY = roach->y - (roachSpeed * roach->rp->sine);
    
    if (RoachInRect(roach, (int)newX, (int)newY, 
		    0, 0, display_width, display_height)) {
	
	roach->x = newX;
	roach->y = newY;

	if (roach->steps-- <= 0) {
	    TurnRoach(roach);
	    roach->steps = RandInt(200);
	}

	for (ii=rx+1; ii<curRoaches; ii++) {
	    r2 = &roaches[ii];
	    if (RoachOverRect(roach, (int)newX, (int)newY,
			r2->intX, r2->intY, r2->rp->width, r2->rp->height)) {
		TurnRoach(roach);
	    }
	}
    }
    else {
	TurnRoach(roach);
    }
}
    
/*
   Draw all roaches.
*/
void
DrawRoaches()
{
    Roach *roach;
    int rx;
    BOOL bResult;
    HBRUSH hOldBrush;
    RECT rRect;
    
    for (rx=0; rx<curRoaches; rx++) {
	roach = &roaches[rx];
	rRect.left = roach->intX;
	rRect.top = roach->intY;
	rRect.right = roach->intX + roach->rp->width;
	rRect.bottom = roach->intY + roach->rp->height;
	if (roach->intX >= 0 && roach->rp != NULL) {
		bResult = InvalidateRect(hDesktopWnd,&rRect ,TRUE);
		if (!bResult) {
			printErrorInfo("InvalidateRect",GetLastError());
		}
	}
    }
    
    hDesktopDC = GetDC(hDesktopWnd);
    hOldBrush = SelectObject(hDesktopDC,hRoachBrush);
    for (rx=0; rx<curRoaches; rx++) {
	roach = &roaches[rx];
	
	if (!roach->hidden) {
	    roach->intX = roach->x;
	    roach->intY = roach->y;
	    roach->rp = &roachPix[roach->index];
	    SelectObject(hRoachDC, roach->rp->hBitmap);
	    BitBlt(hDesktopDC, roach->intX, roach->intY,
		       roach->rp->width,
		       roach->rp->height,
		       hRoachDC,
		       0,
		       0,
		       0xE20746L );
	}
	else {
	    roach->intX = -1;
	}
    }
    SelectObject(hDesktopDC,hOldBrush);
    ReleaseDC(hDesktopWnd, hDesktopDC);
}


DWORD dwCustClrs[16];
BOOL
ChooseNewRoachColor(HWND hwnd, COLORREF *pcrNewColor,COLORREF crOldColor)
{

    CHOOSECOLOR  chsclr;
    BOOL fSetColor = FALSE;

    chsclr.lStructSize = sizeof(CHOOSECOLOR);
    chsclr.hwndOwner = hwnd;
    chsclr.hInstance = GetModuleHandle(NULL);
    chsclr.rgbResult = crOldColor;
    chsclr.lpCustColors = (LPDWORD)dwCustClrs;
    chsclr.lCustData = 0L;
    chsclr.Flags =  CC_RGBINIT;
    chsclr.lpfnHook = (LPCCHOOKPROC)(FARPROC)NULL;
    chsclr.lpTemplateName = (LPSTR)NULL;

   if ( fSetColor = ChooseColor( &chsclr )) {
       *pcrNewColor = chsclr.rgbResult; 
       return TRUE;
   }
   else {
       return FALSE;
   }
}

BOOL bSaveSettings;
BOOL APIENTRY
DlgProc(HWND hDlgWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    // remember: return non-zero if processing the message

    static HBRUSH hTmpSquishBrush;
    static HBRUSH hTmpRoachBrush;
    int DlgCtrID;
    switch (message) {
	case WM_CTLCOLORSTATIC:
	    DlgCtrID = GetDlgCtrlID((HWND)lParam);
	    if (DlgCtrID == IDC_ROACHCOLOR_BOX) {
		SetBkColor((HDC)wParam,crNewRoachColor);
		SetTextColor((HDC)wParam,crNewRoachColor);
		SelectObject((HDC)wParam,hTmpRoachBrush);
		return (BOOL)hTmpRoachBrush;
	    }
	    else if (DlgCtrID == IDC_ROACHGUTSCOLOR_BOX) {
		SetBkColor((HDC)wParam,crNewSquishColor);
		SetTextColor((HDC)wParam,crNewSquishColor);
		SelectObject((HDC)wParam,hTmpSquishBrush);
		return (BOOL)hTmpSquishBrush;
	    }
	    return TRUE;
	case WM_INITDIALOG:
	    crNewRoachColor = crRoachColor;
	    crNewSquishColor = crSquishColor;
	    hTmpRoachBrush = CreateSolidBrush(crRoachColor);
	    hTmpSquishBrush = CreateSolidBrush(crSquishColor);
	    SetDlgItemText(hDlgWnd,IDC_VERSION,NTRoachVersion);
	    SetDlgItemInt(hDlgWnd,IDC_NUMROACHES,maxRoaches,FALSE);
	    SetDlgItemInt(hDlgWnd,IDC_SPEEDROACHES,(int)roachSpeed,FALSE);
	    CheckDlgButton(hDlgWnd,IDC_SAVECONFIGURATION,bSaveSettings);
	    return TRUE;
	case WM_COMMAND:
	    switch(LOWORD(wParam)) {
		case IDOK:
		    DeleteObject(hTmpRoachBrush);
		    DeleteObject(hTmpSquishBrush);
		    newMaxRoaches = GetDlgItemInt(
			hDlgWnd,IDC_NUMROACHES,NULL,0);
		    newSpeed = GetDlgItemInt(
			hDlgWnd,IDC_SPEEDROACHES,NULL,0);
		    if (IsDlgButtonChecked(hDlgWnd,IDC_SAVECONFIGURATION)) {
			bSaveSettings = TRUE;
		    }
		    else {
			bSaveSettings = FALSE;
		    }
		    EndDialog(hDlgWnd,IDOK);
		    return TRUE;
		case IDCANCEL:
		    DeleteObject(hTmpRoachBrush);
		    DeleteObject(hTmpSquishBrush);
		    EndDialog(hDlgWnd,IDCANCEL);
		    return TRUE;
		case IDC_ROACHCOLOR:
		    if (ChooseNewRoachColor(hDlgWnd,&crNewRoachColor,
			    crNewRoachColor)) {
			DeleteObject(hTmpRoachBrush);
			hTmpRoachBrush = CreateSolidBrush(crNewRoachColor);
			bNewRoachColor = TRUE;
			InvalidateRgn(hDlgWnd, NULL, TRUE);
			UpdateWindow(hDlgWnd);
		    }
		    return TRUE;
		case IDC_ROACHGUTCOLOR:
		    if (ChooseNewRoachColor(hDlgWnd,&crNewSquishColor,
			    crNewSquishColor)) {
			DeleteObject(hTmpSquishBrush);
			hTmpSquishBrush = CreateSolidBrush(crNewSquishColor);
			bNewSquishColor = TRUE;
			InvalidateRgn(hDlgWnd, NULL, TRUE);
			UpdateWindow(hDlgWnd);
		    }
		    return TRUE;
		default:
		    break;
	    }
	default:
	    break;
    }
    return 0;
}

void
GetNTRoachSettings()
{
    LONG lResult;
    DWORD dwType;
    DWORD dwValue;
    DWORD dwValueSize;

    lResult = RegQueryValueEx(hRoachKey,
	    szNumRoaches,
	    0,
	    &dwType,
	    (char *)&dwValue,
	    &dwValueSize);
    if (lResult == ERROR_SUCCESS) {
	    maxRoaches = dwValue;
    }
    lResult = RegQueryValueEx(hRoachKey,
	    szSpeedRoaches,
	    0,
	    &dwType,
	    (char *)&dwValue,
	    &dwValueSize);
    if (lResult == ERROR_SUCCESS) {
	    roachSpeed = (float)dwValue;
    }
    lResult = RegQueryValueEx(hRoachKey,
	    szRoachColor,
	    0,
	    &dwType,
	    (char *)&dwValue,
	    &dwValueSize);
    if (lResult == ERROR_SUCCESS) {
	    crRoachColor = dwValue;
    }
    lResult = RegQueryValueEx(hRoachKey,
	    szSquishColor,
	    0,
	    &dwType,
	    (char *)&dwValue,
	    &dwValueSize);
    if (lResult == ERROR_SUCCESS) {
	    crSquishColor = dwValue;
    }
}

void
StoreNewNTRoachSettings()
{
    DWORD dwValue;

    dwValue = newMaxRoaches;
    RegSetValueEx(hRoachKey,
	    szNumRoaches,
	    0,
	    REG_DWORD,
	    (char *)&dwValue,
	    sizeof (DWORD));

    dwValue = (int) newSpeed;
    RegSetValueEx(hRoachKey,
	    szSpeedRoaches,
	    0,
	    REG_DWORD,
	    (char *)&dwValue,
	    sizeof (DWORD));

    dwValue = crNewRoachColor;
    RegSetValueEx(hRoachKey,
	    szRoachColor,
	    0,
	    REG_DWORD,
	    (char *)&dwValue,
	    sizeof (DWORD));

    dwValue = crNewSquishColor;
    RegSetValueEx(hRoachKey,
	    szSquishColor,
	    0,
	    REG_DWORD,
	    (char *)&dwValue,
	    sizeof (DWORD));
}


void
RoachSettings(HWND hwnd)
{
    HBRUSH hOldBrush;
    if (DialogBox(GetModuleHandle(NULL),
		"SettingsDialog", 
		hwnd, DlgProc) == IDOK ) {
	if (bNewSquishColor) {
	    bNewSquishColor = FALSE;
	    crSquishColor = crNewSquishColor;
	    hOldBrush = hSquishBrush;
	    hSquishBrush = CreateSolidBrush(crSquishColor);
	    DeleteObject(hOldBrush);
	}
	if (bNewRoachColor) {
	    bNewRoachColor = FALSE;
	    crRoachColor = crNewRoachColor;
	    hOldBrush = hRoachBrush;
	    hRoachBrush = CreateSolidBrush(crRoachColor);
	    DeleteObject(hOldBrush);
	}
	if ((newSpeed) && (newSpeed != (int)roachSpeed)) {
	    roachSpeed = (float) newSpeed;
	}
	if ((newMaxRoaches) && (newMaxRoaches != maxRoaches)) {
	    SetEvent(hNewNumRoachesEvent);
	}
	if (bSaveSettings) {
	    StoreNewNTRoachSettings();
	}
    }
}

void
SquishState(BOOL bNewState)
{
static BOOL bSquishingState = FALSE;

    if (bNewState == bSquishingState) {
	return;
    }
    bSquishingState = bNewState;
    if (bNewState==TRUE) {
	SetClassLong(hMainWnd,GCL_HICON, (long) hSquishIcon);
	SetCapture(hMainWnd);
    }
    else {
	SetClassLong(hMainWnd,GCL_HICON,(long) hRoachIcon);
	ReleaseCapture();
    }
    FlashWindow(hMainWnd,FALSE);
}

LRESULT CALLBACK 
MainWndProc (HWND hwnd, UINT message, WPARAM wParam,
                              LPARAM lParam)
{

    RECT rRect;
    switch (message) {
	case WM_SYSCOMMAND:
	    wParam &= -16;  /* System requires that low 4 bits be masked */
	    if (wParam == ARM_ROACH_SQUISHER) {
		SquishState(TRUE);
	    }
	    else if (wParam == ROACH_CONFIG) {
		RoachSettings(hwnd);
	    }
	    else if (wParam == HIDDEN) {
	    	ShowWindow(hwnd,SW_HIDE);
	    }
	    else if (wParam == NOT_HIDDEN) {
	    	ShowWindow(hwnd,SW_SHOWMINIMIZED);
	    }
	    break;
	case WM_INITMENUPOPUP:
	    if (HIWORD(lParam)){ /* System Popup */
		    EnableMenuItem(hSystemMenu, SC_MAXIMIZE,MF_GRAYED);
		    EnableMenuItem(hSystemMenu, SC_RESTORE,MF_GRAYED);
		    return 0;
	    }
	    break;
	case WM_QUERYOPEN:
	    return FALSE;  /* We want to stay an Icon */
	case WM_DESTROY:
	    PostQuitMessage(0);
	    return FALSE;
	case WM_CANCELMODE:
	    return FALSE; /*  only slightly antisocial */

	case WM_LBUTTONDOWN:
	case WM_MBUTTONDOWN:
	case WM_RBUTTONDOWN:
	    pMouse.x = (signed short) LOWORD(lParam);
	    pMouse.y = (signed short) HIWORD(lParam);
	    ClientToScreen(hwnd,&pMouse);
	    GetWindowRect(hwnd,&rRect);

	    if (PtInRect(&rRect,pMouse)) {
		SquishState(FALSE);
	    }
	    else {
		SetEvent(hCheckSquishEvent);
	    }
	    break;
	case WM_ACTIVATEAPP:
	    SquishState(FALSE);
	    break;
	default:
	    break;

      }
    return (DefWindowProc(hwnd, message, wParam, lParam));
}

void
RoachMainLoop() {
    int rx;

    DrawRoaches();
    while(TRUE) {
	if (WaitForSingleObject(hNeedCalcEvent,0) == WAIT_OBJECT_0) {
	    CalcRootVisible();
	}
	else if (WaitForSingleObject(hCheckSquishEvent,0) == WAIT_OBJECT_0) {
	    CheckSquish();
	}
	else if (WaitForSingleObject(hNewNumRoachesEvent,0) == WAIT_OBJECT_0) {
	    NewNumRoaches();
	}
	else if (MarkHiddenRoaches()) {
	    for (rx=0; rx<curRoaches; rx++) {
		    if (!roaches[rx].hidden )
			    MoveRoach(rx);
	    }
	    DrawRoaches();
	}
	else {
	    WaitForSingleObject(hNeedCalcEvent, (DWORD) -1);
	    CalcRootVisible();
	}
    }
}

UINT idTimer;
BOOL APIENTRY
PrevDlgProc(HWND hPrevDlgWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    // remember: return non-zero if processing the message

    switch (message) {
	case WM_INITDIALOG:
	    SetTimer(hPrevDlgWnd,idTimer,1000,NULL);
	    return TRUE;
	case WM_TIMER:
		KillTimer(hPrevDlgWnd,idTimer);
		EndDialog(hPrevDlgWnd,IDOK);
		return TRUE;
	case WM_COMMAND:
	    switch(LOWORD(wParam)) {
		case IDOK:
		    EndDialog(hPrevDlgWnd,IDOK);
		    return TRUE;
		default:
		    break;
	    }
	default:
	    break;
    }
    return 0;
}

#ifdef CONSOLE_APP
int 
main()
{
    HINSTANCE hInstance;
    HINSTANCE hPrevInstance;
    int nCmdShow;
#else
int WINAPI 
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
		   LPSTR lpszCmdLine, int nCmdShow)
{
#endif

    RoachMap *rp;
    int rx;
    float angle;
    static char szAppName[] = "NTRoach";
    MSG msg;
    WNDCLASS wndclass;
    BOOL bResult;
    LONG lResult;
    DWORD dwDisp;
    int ax;

#ifdef CONSOLE_APP
    hPrevInstance = 0;
    hInstance = GetModuleHandle(NULL);
    nCmdShow = SW_SHOWDEFAULT; 
#endif

    /* Check to make sure we are running on Windows NT */
    if( GetVersion() & 0x80000000 ) {
	MessageBox(NULL, "Sorry, NTRoach requires Windows NT.",
		          "Error: Windows NT Required to Run",  MB_OK );
	return(0);
    }
    if (hMainWnd = FindWindow("NTRoach","NTRoach")) {
	bResult = SetForegroundWindow(hMainWnd);
	if (bResult) {
	    SendMessage(hMainWnd,WM_SYSCOMMAND,NOT_HIDDEN,0);
	    if (lResult = 
	    	DialogBox(GetModuleHandle(NULL), 
			"Previous", hMainWnd, PrevDlgProc)== -1) {
	    	printErrorInfo("DialogBox",GetLastError());
	    }
	    exit(0);
	}
    	
    }
    lResult = RegCreateKeyEx(HKEY_CURRENT_USER,
			szRoachKey,
			0,
			0,
			0,
			KEY_ALL_ACCESS,
			0,
			&hRoachKey,
			&dwDisp);

    if (lResult == ERROR_SUCCESS) {
	GetNTRoachSettings();	
    }
    else {
	MessageBox(NULL,
		"Can't access Registry. Will use default configuration.",
		"NTRoach",MB_OK);
    }

    srand((int)time((long *)NULL));

    hNeedCalcEvent = CreateEvent(
			NULL /* security attributes */,
			TRUE /* manual reset */,
			TRUE /* initial state */,
			RECALC_EVENT );
    if (!hNeedCalcEvent) {
	printErrorInfo("CreateEvent",GetLastError());
	exit(0);
    }
    SetEvent(hNeedCalcEvent);
    hCheckSquishEvent = CreateEvent(
			    NULL /* security attributes */,
			    TRUE /* manual reset */,
			    FALSE /* initial state */,
			    SQUISH_EVENT);

    hNewNumRoachesEvent = CreateEvent(
			    NULL /* security attributes */,
			    TRUE /* manual reset */,
			    FALSE /* initial state */,
			    NEWNUMROACHES_EVENT);
    signal(SIGTERM,SigHandler);
    signal(SIGINT,SigHandler);
    hDesktopWnd = GetDesktopWindow();
    bResult = GetWindowRect(hDesktopWnd,&rDesktopRect);
    if(!bResult) {
	printErrorInfo("GetWindowRect",GetLastError());
	exit(0);
    }

    display_width = rDesktopRect.right - rDesktopRect.left;
    display_height = rDesktopRect.bottom - rDesktopRect.top;
    center_x = display_width / 2;
    center_y = display_height / 2;
    
    /*
       Create roach pixmaps at several orientations.
    */
    hDesktopDC = GetDC(hDesktopWnd);
    for (ax=0; ax<360; ax+=ROACH_ANGLE) {
	BITMAP bm;
	rx = ax / ROACH_ANGLE;
	angle = (float) (rx * 0.261799387799);
	rp = &roachPix[rx];
	if ((rp->hBitmap = 
	    LoadBitmap(hInstance, MAKEINTRESOURCE(rp->roachBits))) == NULL) {
	    printErrorInfo("LoadBitmap",GetLastError());
		exit(0);
	}
	GetObject(rp->hBitmap, sizeof(BITMAP), (LPSTR) & bm);
	rp->width = bm.bmWidth;
	rp->height = bm.bmHeight;
	rp->sine = (float) sin(angle);
	rp->cosine = (float) cos(angle);
    }
    hRoachDC =  CreateCompatibleDC(hDesktopDC);


    if ((hSquishBitMap =
	  LoadBitmap(hInstance, MAKEINTRESOURCE(squishBits))) == NULL) {
	    printErrorInfo("LoadBitmap",GetLastError());
	    exit(0);
    }
    hSquishDC =  CreateCompatibleDC(hDesktopDC);
    SelectObject(hSquishDC,hSquishBitMap);
    hRoachBrush = CreateSolidBrush(crRoachColor);
    hSquishBrush = CreateSolidBrush(crSquishColor);
    ReleaseDC(hDesktopWnd, hDesktopDC);

    roaches = (Roach *)malloc(sizeof(Roach) * maxRoaches);

    while (curRoaches < maxRoaches) {
	AddRoach();
    }
    hRoachIcon = LoadIcon(hInstance, 
			    MAKEINTRESOURCE(IDI_ROACH45));

    hSquishIcon = LoadIcon(hInstance, 
			    MAKEINTRESOURCE(IDI_SQUISH));

    wndclass.style         = /* WS_MINIMIZE */ 0;
    wndclass.lpfnWndProc   = (WNDPROC)MainWndProc;
    wndclass.cbClsExtra    = 0;
    wndclass.cbWndExtra    = 0;
    wndclass.hInstance     = hInstance;
    wndclass.hIcon         = hRoachIcon;
    wndclass.hCursor       = LoadCursor (NULL, IDC_ARROW);
    wndclass.hbrBackground = GetStockObject (WHITE_BRUSH);
    wndclass.lpszMenuName  = NULL;
    wndclass.lpszClassName = (LPCTSTR) "NTRoach";

    if (!RegisterClass (&wndclass)) {
	  printErrorInfo("RegisterClass",GetLastError());
      return(0);
    }

    if (!(hMainWnd = CreateWindow ("NTRoach", "NTRoach",
			  WS_OVERLAPPEDWINDOW ,
			  CW_USEDEFAULT, CW_USEDEFAULT,
			  CW_USEDEFAULT, CW_USEDEFAULT,
			  NULL, NULL, hInstance, NULL))) {
	return (0);
    }
    bMainWnd = TRUE;

    hSystemMenu = GetSystemMenu(hMainWnd,FALSE);

    AppendMenu(hSystemMenu, 
		MF_SEPARATOR, 0, NULL);

    AppendMenu(hSystemMenu, 
		MF_STRING, HIDDEN, "&Hide");
    AppendMenu(hSystemMenu, 
		MF_STRING, ARM_ROACH_SQUISHER, "Squish ON");
    AppendMenu(hSystemMenu, 
		MF_STRING, ROACH_CONFIG, "Settings...");



    hNTRHook = LoadLibrary(NTRHook);
    if (!hNTRHook) {
	printErrorInfo("LoadLibrary: " NTRHook ,GetLastError());
	exit(0);
    }
    InstallHook =  (int (*)())GetProcAddress (hNTRHook, "InstallHook");
    if (!InstallHook) {
	printErrorInfo("InstallHook - GetProcAddress",GetLastError());
	exit(0);
    }
    UnInstallHook =  (int (*)())GetProcAddress (hNTRHook, "UnInstallHook");
    if (!UnInstallHook) {
	printErrorInfo("InstallHook - GetProcAddress",GetLastError());
	exit(0);
    }
    lResult = InstallHook();
    if (!lResult) {
	printErrorInfo("InstallHook - NTRHook Error",lResult);
	exit(0);
    }

#ifdef CONSOLE_APP
    ShowWindow (hMainWnd,SW_SHOWMINIMIZED);
#else
    ShowWindow (hMainWnd, 
    	strcmp(lpszCmdLine,"-hidden")?SW_SHOWMINIMIZED :SW_HIDE);
#endif

    CreateThread (NULL, 0, 
		(LPTHREAD_START_ROUTINE) RoachMainLoop, 
		NULL, 0, &dwThreadId);

    SetPriorityClass(GetCurrentProcess(),IDLE_PRIORITY_CLASS);


    while (GetMessage (&msg, NULL, 0, 0)) {
	TranslateMessage (&msg);
	DispatchMessage  (&msg);
    }

    bMainWnd = FALSE;
    lResult = UnInstallHook();
    if (lResult) {
    	printErrorInfo("UnInstallHook",lResult);
    }
    InvalidateRect(hDesktopWnd,&rDesktopRect,TRUE);
    return (msg.wParam);
}

