//**************************************************************************
//
//  CCMPRS -- Conditional File Compression Utility for Windows NT
//
//  Written by Robert Epps
//
//  This program and its source are in the public domain.  If you find any
//  bugs, or add features to this program, I'd like to hear about them!
//  Please write to me at:
//
//  Internet:  robepps@valleynet.com
//  CIS:       72560,3353
//  AOL:       RobertE49
//
//**************************************************************************
#define STRICT
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <winioctl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>


//**************************************************************************
// Global variables representing the various options.
//**************************************************************************
char   szFileSpec[_MAX_PATH];       // Path and filename of file(s) to examine
BOOL   bRecurse    = FALSE;         // TRUE = recurse into subdirectories
BOOL   bDecompress = FALSE;         // TRUE = do decompression
BOOL   bQuiet      = FALSE;         // TRUE = don't display any status info
double threshold   = 0.75;          // Allow compression if (compressed size) <= (uncompressed size) * threshold
double MinFileSize = 0;             // Minimum uncompressed size for file to be compressed

char *szIgnoreExts;                 // Points to string containing extensions of files to ignore (command-delimited)
char *szExts;                       // Points to copy of above string, used for strtok() parsing

UINT nCompressed;                   // # of files compressed
UINT nDecompressed;                 // # of files decompressed
double sizechange;                  // Total size change after files compressed/decompressed

char szTemp[_MAX_PATH];


//**************************************************************************
//  Routine to display helpful instructions.
//**************************************************************************
void DisplayUsage(void)
{
    printf("CCMPRS -- Conditional File Compression/Decompression Utility\n\n");
    printf("Command line format is:\n\n");
    printf("  ccmprs filespec [options]\n\n");
    printf("where filespec is the path and filename (wildcards allowed) of the\n");
    printf("file(s) to be examined.  The options are:\n\n");
    printf("  -s  : Examine files in subdirectories as well as the specified directory.\n");
    printf("  -d  : Decompress any compressed files encountered that do not meet the\n");
    printf("        specified conditions for compression.\n");
    printf("  -t# : Compress file only if resulting file size is # percent or less of\n");
    printf("        the uncompressed size.  If this option is not specified, the default\n");
    printf("        is 75 percent.\n");
    printf("  -m# : Compress file only if its uncompressed size is at least # bytes.  If\n");
    printf("        this option is not specified, the default is 0.\n");
    printf("  -xa[,b[,c...]] : Ignore files with the specified extension(s).  For example,\n");
    printf("        -xzip,gif would ignore ZIP and GIF files.\n");
    printf("  -q  : Operate quietly, with no status info displayed.\n\n");
    //printf("For example:  ccmprs d:\\letters\\*.txt -s -t66.6 -m500000\n\n");
    //printf("This would examine all uncompressed files with file extension .txt, in the\n");
    //printf("directory d:\\letters and all of its subdirectories.  Each file at least 500000\n");
    //printf("bytes in length is compressed.  If the size of the resulting compressed file is\n");
    //printf("66.6%% or less of the uncompressed size, the file is left compressed, otherwise\n");
    //printf("it is restored to its original uncompressed state.\n");
}


//**************************************************************************
//
//  Routine returns TRUE if volume containing the specified path supports
//  file compression, FALSE if not.  The path is assumed to be a full path,
//  either UNC or DOS format.
//
//**************************************************************************
BOOL DriveIsCompressible(LPSTR lpszFileSpec)
{
    char  szRoot[4];
    DWORD dwFlags;
    BOOL  bRet = FALSE;

    do
    {
        // Extract the root directory.
        strncpy(szRoot, lpszFileSpec, 3);
        szRoot[3] = 0;

        // Get information on the volume.
        if (!GetVolumeInformation(szRoot, NULL, 0, NULL, NULL, &dwFlags, NULL, NULL))
            break;

        // Return TRUE if volume supports file compression.
        if (dwFlags & FS_FILE_COMPRESSION)
            bRet = TRUE;

    } while (FALSE);

    return bRet;
}


//**************************************************************************
//  Routine to read the command line parameters and set the various global
//  variables accordingly.
//**************************************************************************
BOOL GetParameters(int ac, char *av[])
{
    int i;

    // For each item in the command line...
    for (i = 1; i < ac; i++)
    {
        // Process option or file specification.
        if (*av[i] == '-')
        {
            switch (*(av[i] + 1))
            {
                case 's':               // The "look at subdirectories" option
                case 'S':
                    bRecurse = TRUE;
                    break;

                case 'q':               // The "be quiet!" option
                case 'Q':
                    bQuiet = TRUE;
                    break;

                case 'd':               // The "decompress" option
                case 'D':
                    bDecompress = TRUE;
                    break;

                case 'm':               // The "minimum size" option
                case 'M':
                    MinFileSize = atof(av[i] + 2);
                    if (MinFileSize < 0)
                    {
                        printf("Invalid minimum file size of %.0lf bytes specified.\n", MinFileSize);
                        return FALSE;
                    }
                    break;

                case 't':               // The "compression threshold" option
                case 'T':
                    threshold = atof(av[i] + 2);
                    if (threshold <= 0 || threshold > 100.0)
                    {
                        printf("Invalid compression threshold of %lf%% specified.\n", threshold);
                        return FALSE;
                    }
                    threshold /= 100.0;
                    break;

                case 'x':               // The "ignore file extensions" option
                case 'X':
                    if (*(av[i] + 2))
                    {
                        if (szIgnoreExts)
                        {
                            // Append extensions to current list of extensions.
                            char *p = (char *)realloc(szIgnoreExts, strlen(szIgnoreExts) + strlen(av[i] + 2) + 2);
                            if (p)
                            {
                                szIgnoreExts = p;
                                strcat(szIgnoreExts, ",");
                                strcat(szIgnoreExts, av[i] + 2);
                                strlwr(szIgnoreExts);
                            }
                        }
                        else
                        {
                            // Make new list of extensions.
                            szIgnoreExts = (char *)malloc(strlen(av[i] + 2) + 1);
                            if (szIgnoreExts)
                            {
                                strcpy(szIgnoreExts, av[i] + 2);
                                strlwr(szIgnoreExts);
                            }
                        }

                    }
                    break;

                default:
                    printf("The -%c option is not a valid option.\n", *(av[i] + 1));
                    return FALSE;
            }
        }
        else
        {
            _fullpath(szFileSpec, av[i], _MAX_PATH);
        }
    }

    // Check for file name.
    if (!szFileSpec[0])
    {
        printf("No file name was specified.\n");
        return FALSE;
    }

    // Make sure that the drive supports compression.
    if (!DriveIsCompressible(szFileSpec))
    {
        printf("The specified volume does not support file-level compression.\n");
        return FALSE;
    }

    // Allocate space for copy of file extensions string.
    if (szIgnoreExts)
    {
        szExts = (char *)malloc(strlen(szIgnoreExts) + 1);
        if (!szExts)
        {
            free(szIgnoreExts);
            szIgnoreExts = NULL;
        }
    }

    return TRUE;
}


//**************************************************************************
//**************************************************************************
BOOL SetFileCompression(LPSTR lpszFile, BOOL bCompress)
{
    HANDLE hFile;
    USHORT uMode;
    DWORD  dwAttributes;
    BOOL   bRet = FALSE;

    do
    {
        // Get the file's current compression state.
        dwAttributes = GetFileAttributes(lpszFile);
        if (dwAttributes == 0xFFFFFFFF)
            break;

        // If the file is already in the desired state, return now.
        if (!bCompress == !(dwAttributes & FILE_ATTRIBUTE_COMPRESSED))
        {
            bRet = TRUE;
            break;
        }

        // Open the file.
        hFile = CreateFile(szTemp, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_ATTRIBUTE_READONLY, NULL);
        if (hFile == INVALID_HANDLE_VALUE)
            break;

        // Set the file's compression state.
        uMode = bCompress ? COMPRESSION_FORMAT_DEFAULT : COMPRESSION_FORMAT_NONE;
        bRet = DeviceIoControl(hFile, FSCTL_SET_COMPRESSION, &uMode, sizeof(uMode), NULL, 0, &dwAttributes, NULL);

        // Close the file.
        CloseHandle(hFile);

    } while (FALSE);

    return bRet;
}


//**************************************************************************
//  Main routine to do conditional file compression/decompression.
//**************************************************************************
int DoIt(LPSTR szFile)
{
    LPWIN32_FIND_DATA pFind  = NULL;
    HANDLE            hFind  = INVALID_HANDLE_VALUE;
    HANDLE            hFile  = INVALID_HANDLE_VALUE;
    int               nRet   = 0;
    int               cSlash = '\\';
    LPSTR             pszPath, p, pszFileName;
    DWORD             dwErr;
    static char       szExt[_MAX_PATH];

    do
    {
        // Allocate file-find structure.
        pFind = (LPWIN32_FIND_DATA)malloc(sizeof(WIN32_FIND_DATA));
        if (!pFind)
        {
            printf("Out of memory!\n");
            nRet = 1;
            break;
        }

        // Find start of file name within path, and get the slash character
        // being used.
        p = szFile + strlen(szFile);
        while (p != szFile && *p != '\\' && *p != '/' && *p != ':')
            p--;
        if (*p == '\\' || *p == '/' || *p == ':')
        {
            if (*p++ == '/')
                cSlash = '/';
        }
        pszFileName = p;

        // If we are recursing, scan for directories and recurse into any that we find.
        if (bRecurse)
        {
            char *szSaveFileName = (char *)malloc(strlen(pszFileName) + 1);
            if (szSaveFileName)
            {
                // Save file name portion of file specification, and temporarily
                // replace it with "*.*".
                strcpy(szSaveFileName, pszFileName);
                strcpy(pszFileName, "*.*");

                // For each directory...
                hFind = FindFirstFile(szFile, pFind);
                if (hFind != INVALID_HANDLE_VALUE)
                {
                    do
                    {
                        if ((pFind->dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) && pFind->cFileName[0] != '.')
                        {
                            // Create new filespec with directory name appended to directory portion.
                            pszPath = (LPSTR)malloc(strlen(szFile) + strlen(pFind->cFileName) + 6);
                            if (!pszPath)
                            {
                                printf("Out of memory!\n");
                                nRet = 1;
                                break;
                            }
                            strcpy(pszPath, szFile);
                            p = pszPath + (pszFileName - szFile);
                            strcpy(p, pFind->cFileName);
                            p += strlen(p);
                            *p++ = cSlash;
                            strcpy(p, szSaveFileName);

                            // Do the recursion.
                            nRet = DoIt(pszPath);

                            // Clean up.
                            free(pszPath);

                            // Bail now on error.
                            if (nRet)
                                break;
                        }

                    } while (FindNextFile(hFind, pFind));

                    // Clean up.
                    FindClose(hFind);
                }

                // Restore original file name.
                strcpy(pszFileName, szSaveFileName);
                free(szSaveFileName);
            }
        }

        // Start search for files.
        hFind = FindFirstFile(szFile, pFind);
        if (hFind == INVALID_HANDLE_VALUE)
        {
            //printf("The specified file path, %s, is invalid.\n", szFile);
            //nRet = 1;
            break;
        }

        // For each file...
        do
        {
            // Skip directories.
            if (pFind->dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
                continue;

            // Ignore the file if its extension is in the "ignore extensions" list.
            if (szIgnoreExts)
            {
                for (p = pFind->cFileName + strlen(pFind->cFileName); p != pFind->cFileName && *p != '.'; p--) ;
                if (*p == '.')
                {
                    p++;
                    if (*p)
                    {
                        strcpy(szExt, p);
                        strlwr(szExt);
                        strcpy(szExts, szIgnoreExts);
                        p = strtok(szExts, " ,");
                        while (p)
                        {
                            if (!strcmp(p, szExt))
                                break;
                            p = strtok(NULL, " ,");
                        }
                        if (p)
                            continue;
                    }
                }
            }

            // Make full path and file name.
            strcpy(szTemp, szFile);
            strcpy(szTemp + (pszFileName - szFile), pFind->cFileName);

            // Get the file's compressed and uncompressed file sizes.
            DWORD dwFileSizeHi, dwCompSizeHi;
            hFile = CreateFile(szTemp, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL | FILE_ATTRIBUTE_READONLY, NULL);
            if (hFile == INVALID_HANDLE_VALUE)
                continue;
            DWORD dwFileSizeLo = GetFileSize(hFile, &dwFileSizeHi);
            CloseHandle(hFile);
            if (dwFileSizeLo == 0xFFFFFFFF && (dwErr = GetLastError()) != NO_ERROR)
                continue;
            DWORD dwCompSizeLo = GetCompressedFileSize(szTemp, &dwCompSizeHi);
            if (dwCompSizeLo == 0xFFFFFFFF && (dwErr = GetLastError()) != NO_ERROR)
                continue;
            double filesize = (double)dwFileSizeHi * 4294967296.0 + (double)dwFileSizeLo;
            double compsize = (double)dwCompSizeHi * 4294967296.0 + (double)dwCompSizeLo;

            // Compress or decompress the file depending on compression state and flags.
            if (compsize == filesize)
            {
                // File is not compressed.  Skip it if it does not meet minimum size
                // requirement for compression.
                if (filesize < MinFileSize)
                    continue;

                // Compress the file.
                if (!SetFileCompression(szTemp, TRUE))
                    continue;

                // Get the new compressed size.
                dwCompSizeLo = GetCompressedFileSize(szTemp, &dwCompSizeHi);
                if (dwCompSizeLo == 0xFFFFFFFF && GetLastError() != NO_ERROR)
                {
                    SetFileCompression(szTemp, FALSE);
                    continue;
                }
                compsize = (double)dwCompSizeHi * 4294967296.0 + (double)dwCompSizeLo;

                // If the file did not compress enough, decompress it.
                if (compsize > filesize * threshold)
                    SetFileCompression(szTemp, FALSE);
                else
                {
                    sizechange += compsize - filesize;
                    nCompressed++;
                    if (!bQuiet)
                        printf("%s compressed.\n", szTemp);
                }
            }
            else
            {
                // File is compressed.  If the -d flag was not specified, just skip it.
                if (!bDecompress)
                    continue;

                // If the compression is not enough, or the file's original size is not
                // big enough, decompress the file.
                if (compsize > filesize * threshold)
                {
                    SetFileCompression(szTemp, FALSE);
                    if (!bQuiet)
                        printf("%s decompressed.\n", szTemp);
                    nDecompressed++;
                    sizechange += filesize - compsize;
                }
                else if (filesize < MinFileSize)
                    SetFileCompression(szTemp, FALSE);
            }

        } while (FindNextFile(hFind, pFind));

    } while (FALSE);

    // Clean up.
    if (hFind != INVALID_HANDLE_VALUE)
        FindClose(hFind);
    if (pFind)
        free(pFind);

    return nRet;
}


//**************************************************************************
//  The main entry point, the head honcho, the Big Cheeze...
//**************************************************************************
int main(int ac, char *av[])
{
    int nRet;

    // If no parameters are specified, display usage instructions.
    if (ac < 2)
    {
        DisplayUsage();
        return 1;
    }

    // Read command line parameters.
    if (!GetParameters(ac, av))
        return 1;

    // Do the compression/decompression.
    nRet = DoIt(szFileSpec);

    // Report results.
    printf("\n%d file(s) compressed, %d file(s) decompressed, ", nCompressed, nDecompressed);
    if (sizechange <= 0)
        printf("%.0lf bytes more free space.\n", -sizechange);
    else
        printf("%.0lf bytes less free space.\n", sizechange);

    // Clean up.
    if (szExts)
        free(szExts);
    if (szIgnoreExts)
        free(szIgnoreExts);

    return nRet;
}
