//@doc		VIEW
//@module	VIEW.H - View inline declarations |
//	
//	This file contains inline definitions for the view classes.
//		
//	Only very small members are selected for inline code.
//	See the class listing for a description of these functions.
//	
//@devnote	1.0 beta
//@normal	Copyright <cp> 1996 Meta Four Software. All rights reserved.

/////////////////////////////////////////////////////////////////////////////
// c4_View

	//@mfunc Destructor.
d4_inline c4_View::~c4_View ()
{
	d4_assertThis;
	
	_DecSeqRef();
}

	//@mfunc Returns the number of entries.
	//@rdesc A non-negative integer.
d4_inline int c4_View::GetSize() const
{
	d4_assertThis;
	d4_assert(_seq);
	
	return _seq->Size();
}

	//@mfunc Returns highest index (size - 1).
	//@rdesc One less than the number of entries.
	// If there are no entries, the value -1 is returned.
d4_inline int c4_View::GetUpperBound() const
{
	d4_assertThis;
	
	return GetSize() - 1;
}

	//@mfunc Changes the size of this view.
d4_inline void c4_View::SetSize(int newSize_, int growBy_)
{
	d4_assertThis;
	d4_assert(_seq);
	
	_seq->Resize(newSize_, growBy_);
}

	//@mfunc Removes all entries (sets size to zero).
d4_inline void c4_View::RemoveAll()
{
	d4_assertThis;
	
	SetSize(0);
}

	//@mfunc Returns a reference to specified entry.
	//@rdesc A reference to the specified row in the view.
d4_inline c4_RowRef c4_View::GetAt(int index_) const
{
	d4_assertThis;
	d4_assert(_seq);
	
	return * c4_Cursor (*_seq, index_);
}

	//@mfunc Shorthand for <mf c4_View::GetAt>.
	//@rdesc A reference to the specified row in the view.
d4_inline c4_RowRef c4_View::operator[] (int index_) const
{
	d4_assertThis;
	
	return GetAt(index_);
}
    
	//@mfunc Changes the value of the specified entry.
d4_inline void c4_View::SetAt(int index_, c4_RowRef newElem_)
{
	d4_assertThis;
	d4_assert(_seq);
	
	_seq->SetAt(index_, &newElem_);
}

	//@mfunc Element access, for use as RHS or LHS.
	//@rdesc A reference to the specified row in the view.
d4_inline c4_RowRef c4_View::ElementAt(int index_)
{
	d4_assertThis;
	d4_assert(_seq);
	
	return GetAt(index_);
}

	//@mfunc Inserts one or more copies of an entry.
d4_inline void c4_View::InsertAt(int index_, c4_RowRef newElem_, int count_)
{
	d4_assertThis;
	d4_assert(_seq);
	
	_seq->InsertAt(index_, &newElem_, count_);
}

	//@mfunc Removes entries starting at the given index.
d4_inline void c4_View::RemoveAt(int index_, int count_)
{
	d4_assertThis;
	d4_assert(_seq);
	
	_seq->RemoveAt(index_, count_);
}

	//@mfunc Returns the number of properties.
	//@rdesc A non-negative integer.
d4_inline int c4_View::NumProperties() const
{
	d4_assertThis;
	d4_assert(_seq);
	
	return _seq->NumHandlers();
}

	//@mfunc Returns the N-th property.
	//@rdesc A reference to the specified property.
d4_inline int c4_View::NthProperty(int index_) const
{
	d4_assertThis;
	d4_assert(_seq);
	
	return _seq->NthProperty(index_);
}

	//@mfunc Finds a property.
	//@rdesc The index of the property, or -1 of it was not found.
d4_inline int c4_View::FindProperty(int propId_)
{
	d4_assertThis;
	d4_assert(_seq);
	
	return _seq->PropIndex(propId_);
}

d4_inline c4_String c4_View::Describe() const
{
	d4_assertThis;
	d4_assert(_seq);
	
	return _seq->Describe();
}

d4_inline c4_View c4_View::Sort() const
{
	d4_assertThis;
	
	return SortOn(*this);
}

d4_inline void c4_View::_IncSeqRef()
{
	d4_assertThis;
	d4_assert(_seq);
	
//	TRACE("IncSeqRef() %08lX\n", _seq);
	
	_seq->IncRef();
}

d4_inline void c4_View::_DecSeqRef()
{
	d4_assertThis;
	d4_assert(_seq);
	
//	TRACE("DecSeqRef() %08lX\n", _seq);
	
	_seq->DecRef();
}

/////////////////////////////////////////////////////////////////////////////
// c4_Cursor

d4_inline c4_Cursor::c4_Cursor (c4_Sequence& seq_, int index_)
	: _seq (&seq_), _index (index_)
{
}

d4_inline c4_RowRef c4_Cursor::operator* () const
{
	d4_assertThis;
	d4_assert(_seq);
	
	return *this;
}

d4_inline c4_Cursor& c4_Cursor::operator++ ()
{
	d4_assertThis;
	
	++_index;
	return *this;
}

d4_inline c4_Cursor c4_Cursor::operator++ (int)
{
	d4_assertThis;
	d4_assert(_seq);
	
	return c4_Cursor (*_seq, _index++);
}

d4_inline c4_Cursor& c4_Cursor::operator-- ()
{
	d4_assertThis;
	
	--_index;
	return *this;
}

d4_inline c4_Cursor c4_Cursor::operator-- (int)
{
	d4_assertThis;
	d4_assert(_seq);
	
	return c4_Cursor (*_seq, _index--);
}

d4_inline c4_Cursor& c4_Cursor::operator+= (int offset_)
{
	d4_assertThis;
	
	_index += offset_;
	return *this;
}

d4_inline c4_Cursor& c4_Cursor::operator-= (int offset_)
{
	d4_assertThis;
	
	_index -= offset_;
	return *this;
}

d4_inline c4_Cursor c4_Cursor::operator- (int offset_) const
{
	d4_assertThis;
	d4_assert(_seq);
	
	return c4_Cursor (*_seq, _index - offset_);
}

d4_inline int c4_Cursor::operator- (c4_Cursor cursor_) const
{
	d4_assertThis;
	d4_assert(_seq == cursor_._seq);
	
	return _index - cursor_._index;
}

d4_inline c4_Cursor operator+ (c4_Cursor cursor_, int offset_)
{
	return c4_Cursor (*cursor_._seq, cursor_._index + offset_);
}

d4_inline c4_Cursor operator+ (int offset_, c4_Cursor cursor_)
{
	return cursor_ + offset_;
}

d4_inline bool operator== (c4_Cursor a_, c4_Cursor b_)
{
	return a_._seq == b_._seq && a_._index == b_._index;
}

d4_inline bool operator!= (c4_Cursor a_, c4_Cursor b_)
{
	return !(a_ == b_);
}

d4_inline bool operator< (c4_Cursor a_, c4_Cursor b_)
{
	return a_._seq < b_._seq ||
			a_._seq == b_._seq && a_._index < b_._index;
}

d4_inline bool operator> (c4_Cursor a_, c4_Cursor b_)
{
	return b_ < a_;
}

d4_inline bool operator<= (c4_Cursor a_, c4_Cursor b_)
{
	return !(b_ < a_);
}

d4_inline bool operator>= (c4_Cursor a_, c4_Cursor b_)
{                     
	return !(a_ < b_);
}

/////////////////////////////////////////////////////////////////////////////
// c4_RowRef

d4_inline c4_RowRef::c4_RowRef (c4_Cursor cursor_)
	: _cursor (cursor_)
{
}

d4_inline c4_RowRef c4_RowRef::operator= (const c4_RowRef& rowRef_)
{
	d4_assertThis;
	d4_assert(_cursor._seq);
	
	if (_cursor != rowRef_._cursor)
		_cursor._seq->SetAt(_cursor._index, &rowRef_);
	
	return *this;
}

d4_inline c4_Cursor c4_RowRef::operator& () const
{
	d4_assertThis;
	
	return _cursor;
}

d4_inline c4_View c4_RowRef::Container() const
{
	d4_assertThis;
	d4_assert(_cursor._seq);
	
	return _cursor._seq;
}

d4_inline bool operator== (c4_RowRef a_, c4_RowRef b_)
{
	d4_assert((&a_)._seq);
	
	return (&a_)._seq->Compare((&a_)._index, &b_) == 0;
}               

d4_inline bool operator!= (c4_RowRef a_, c4_RowRef b_)
{
	return !(a_ == b_);
}

d4_inline bool operator< (c4_RowRef a_, c4_RowRef b_)
{
	d4_assert((&a_)._seq);
	
	return (&a_)._seq->Compare((&a_)._index, &b_) < 0;
}               

d4_inline bool operator> (c4_RowRef a_, c4_RowRef b_)
{
	return b_ < a_;
}

d4_inline bool operator<= (c4_RowRef a_, c4_RowRef b_)
{
	return !(b_ < a_);
}

d4_inline bool operator>= (c4_RowRef a_, c4_RowRef b_)
{                     
	return !(a_ < b_);
}

/////////////////////////////////////////////////////////////////////////////
// c4_Property

d4_inline c4_Property::~c4_Property ()
{
	d4_assertThis;

	Refs(-1);
}

d4_inline int c4_Property::GetId() const
{
	d4_assertThis;

	return _id;
}

d4_inline c4_Reference c4_Property::operator() (c4_RowRef rowRef_) const
{
	d4_assertThis;

	return c4_Reference (rowRef_, GetId());
}

/////////////////////////////////////////////////////////////////////////////
// c4_IntProp

	//@mfunc Constructs a new integer property.
	//@parm const char* | name_ | the name of this property
d4_inline c4_IntProp::c4_IntProp (const char* name_) 
	: c4_Property ('I', name_)
{
}

d4_inline c4_Row c4_IntProp::operator[] (long value_)
{
	d4_assertThis;

	c4_Row row;
	operator() (row) = value_;
	return row;
}
	
d4_inline c4_IntRef c4_IntProp::operator() (c4_RowRef rowRef_)
{
	d4_assertThis;

	return c4_Reference (rowRef_, GetId());
}

/////////////////////////////////////////////////////////////////////////////
// c4_FloatProp

	//@mfunc Constructs a new floating point property.
	//@parm const char* | name_ | the name of this property
d4_inline c4_FloatProp::c4_FloatProp (const char* name_) 
	: c4_Property ('F', name_)
{
}

d4_inline c4_Row c4_FloatProp::operator[] (double value_)
{
	d4_assertThis;

	c4_Row row;
	operator() (row) = value_;
	return row;
}
	
d4_inline c4_FloatRef c4_FloatProp::operator() (c4_RowRef rowRef_)
{
	d4_assertThis;

	return c4_Reference (rowRef_, GetId());
}

/////////////////////////////////////////////////////////////////////////////
// c4_StringProp

	//@mfunc Constructs a new string property.
	//@parm const char* | name_ | the name of this property
d4_inline c4_StringProp::c4_StringProp (const char* name_) 
	: c4_Property ('S', name_)
{
}
	
d4_inline c4_Row c4_StringProp::operator[] (const char* value_)
{
	d4_assertThis;

	c4_Row row;
	operator() (row) = value_;
	return row;
}
	
d4_inline c4_StringRef c4_StringProp::operator() (c4_RowRef rowRef_)
{
	d4_assertThis;

	return c4_Reference (rowRef_, GetId());
}

/////////////////////////////////////////////////////////////////////////////
// c4_ViewProp

	//@mfunc Constructs a new view property.
	//@parm const char* | name_ | the name of this property
d4_inline c4_ViewProp::c4_ViewProp (const char* name_)
	: c4_Property ('V', name_)
{
}
	
d4_inline c4_Row c4_ViewProp::operator[] (const c4_View& value_)
{
	d4_assertThis;

	c4_Row row;
	operator() (row) = value_;
	return row;
}
	
d4_inline c4_ViewRef c4_ViewProp::operator() (c4_RowRef rowRef_)
{
	d4_assertThis;

	return c4_Reference (rowRef_, GetId());
}

/////////////////////////////////////////////////////////////////////////////
// c4_Reference

d4_inline c4_Reference::c4_Reference (c4_RowRef rowRef_, int propId_)
	: _cursor (&rowRef_), _propId (propId_)
{
}

d4_inline bool c4_Reference::GetData(c4_Bytes& buf_) const
{
	d4_assertThis;
	d4_assert(_cursor._seq);
    
	return _cursor._seq->Get(_cursor._index, _propId, buf_);
}

d4_inline void c4_Reference::SetData(const c4_Bytes& buf_) const
{
	d4_assertThis;
	d4_assert(_cursor._seq);
    
	_cursor._seq->Set(_cursor._index, _propId, buf_);
}

/////////////////////////////////////////////////////////////////////////////
// c4_IntRef

d4_inline c4_IntRef::c4_IntRef (const c4_Reference& value_)
	: c4_Reference (value_)
{
}

/////////////////////////////////////////////////////////////////////////////
// c4_FloatRef

d4_inline c4_FloatRef::c4_FloatRef (const c4_Reference& value_)
	: c4_Reference (value_)
{
}

/////////////////////////////////////////////////////////////////////////////
// c4_StringRef

d4_inline c4_StringRef::c4_StringRef (const c4_Reference& value_)
	: c4_Reference (value_)
{
}

/////////////////////////////////////////////////////////////////////////////
// c4_ViewRef

d4_inline c4_ViewRef::c4_ViewRef (const c4_Reference& value_)
	: c4_Reference (value_)
{
}

/////////////////////////////////////////////////////////////////////////////
