//@doc		TABLE
//@module	TABLE.H - Table declarations |
//	
//	This file contains the declaration of the table classes.
//	
//@devnote	1.0 beta
//@normal	Copyright <cp> 1996 Meta Four Software. All rights reserved.

#ifndef __K4TABLE_H__
#define __K4TABLE_H__

#ifndef __K4CONF_H__
	#error Please include "k4conf.h" before this header file
#endif

/////////////////////////////////////////////////////////////////////////////
// Declarations in this file
	
	class c4_Bytes;						// used to pass around generic data
	class c4_Column;					// a column in a table
	class c4_Table;						// a table of data

	class c4_Field;                     // not defined here
	class c4_Streamer;					// not defined here
	class c4_Persist;					// not defined here
    class c4_TableEntry;				// not defined here
    class c4_Sequence;					// not defined here
    class c4_HandlerSeq;				// not defined here

/////////////////////////////////////////////////////////////////////////////
//@class These fellows know how to clean up their act when they go away.
//
// These objects are used to pass around untyped data without concerns about
// clean-up. They know whether the bytes need to be de-allocated when these
// objects go out of scope.

class d4_export c4_Bytes
{
public:
	//@cmember Constructs an empty object.
	c4_Bytes ();
	//@cmember Constructs an object with contents, optionally as a copy.
	c4_Bytes (const void*, int, bool);
	//@cmember Copy constructor.
	c4_Bytes (const c4_Bytes&);
	//@cmember Destructor.
	~c4_Bytes ();
	
	//@cmember Assignment, this may make a private copy of contents.
	c4_Bytes& operator= (const c4_Bytes&);
	//@cmember Swaps the contents and ownership of two byte objects.
    void Swap(c4_Bytes&);
    
	//@cmember Returns the number of bytes of its contents.
	int Size() const;
	//@cmember Returns a pointer to the contents.
	const uchar* Contents() const;
	
	//@cmember Defines contents as a freshly allocated buffer of given size.
	uchar* SetBuffer(int);
	//@cmember Allocates a buffer and fills its contents with zero bytes.
	uchar* SetBufferClear(int);

private:
	void _MakeCopy();
	void _LoseCopy();
	
	const uchar* _contents;
	int _size;
	bool _copy;
};

/////////////////////////////////////////////////////////////////////////////
//@class Objects of this class define the column contents of tables.
//@xref <c c4_Table>

class d4_export c4_Column
{
public: //@access Public members
	//@cmember Provides access to the contents if loaded.
	uchar* Contents() const;

	void SetPointer(uchar*, int);
	uchar* SetBuffer(int);
	uchar* SetBufferClear(int);

	//@cmember Returns the number of bytes as stored on disk.
	ulong Size() const;
	//@cmember Sets a flag to save this column on commit.
	void SetDirty();
	//@cmember Returns true if contents needs to be saved.
	bool IsDirty() const;

		// for DUMP.CPP
    ulong Position() const { return _position; }

	uchar* LoadNow(c4_Persist&);
	bool SaveNow(c4_Streamer&);

private:
	c4_Column ();
	~c4_Column ();
	
	uchar* _pointer;
	ulong _position;
	ulong _size;
	
	friend class c4_Table;
};

/////////////////////////////////////////////////////////////////////////////
//@class Tables organize the data and know their own structure and context.

class d4_export c4_Table
{
public: //@access Public members
//@group Construction / destruction / initialization
	//@cmember Constructs a new object.
	c4_Table ();
	//@cmember Destructor.
	~c4_Table ();

	//@cmember Initializes this table as a root table.
	void DefineRoot(const char*, c4_Persist* =0);
	//@cmember Initializes this table as a nested table.
	void DefineSub(c4_Field&, c4_Table&, int);

//@group Table properties
	//@cmember Returns the associated storage, if any.
	c4_Persist* Persist() const;
	//@cmember Returns the repeating field definition.
	c4_Field& Definition() const;
	//@cmember Returns a reference to the owner of this table.
	c4_Table& Owner() const;
	//@cmember Returns the row index in the owning table.
	int OwnerRow() const;
	//@cmember Returns the column index in the owning table.
	int OwnerColumn() const;
    
//@group Table dimensions
	//@cmember Returns the number of rows in this table.
    int NumRows() const;
	//@cmember Returns the number of columns in this table.
    int NumColumns() const;

//@group Access to subfields
	//@cmember Returns the field definition of a subcolumn.
	c4_Field& Field(int) const;
	//@cmember Returns a specific columns of this table.
	c4_Column& Column(int) const;
	//@cmember Returns true if a column consists of subtables.
    bool IsNested(int) const;
	//@cmember Returns the N-th subtable of a subcolumn, creates if necessary.
    c4_Table& SubTable(int, int);
    
//@group Table persistence
	//@cmember Saves current table state to file.
	void Commit(c4_Streamer&, int =0);
	//@cmember Loads table state from file.
	void Prepare(c4_Streamer&);
	//@cmember Loads one column on demand.
    uchar* LoadColumn(int) const;

	void SetNumRows(int);
	void FixOwnerRows(int, int);

	c4_HandlerSeq& Sequence();
	void ReplaceEntry(int colNum_, int index_, c4_Sequence* seq_);

	static void _DumpAll(const c4_Table*, int =1, int =0);

private:
	c4_Table (const c4_Table&);			// not implemented
	void operator= (const c4_Table&);	// not implemented

	c4_TableEntry* SubEntries(int) const;
	void InitSequence();
	void SetOwnerRow(int);

	void PrepareColumn(int, c4_Streamer&);
	void PrepareSubTables(int, c4_Streamer&);

	c4_PtrArray _columns;
	c4_Persist* _persist;
	c4_Field* _field;	
	c4_Table* _owner;
	int _ownerRow;
	c4_HandlerSeq* _sequence;
};

/////////////////////////////////////////////////////////////////////////////

#if q4_INLINE
	#include "k4table.inl"
#endif

/////////////////////////////////////////////////////////////////////////////

#endif
