// Copyright (c) 1994, William Wagner
// All Rights reserved.
//
// This source is a portion of a shareware program.  It may be distributed
// only in its entirety.  The copyright statements must be included with any 
// reproduction of this source.
// 

// cassedoc.cpp : implementation of the CCassetteDoc class
//

#include "stdafx.h"
#include "cassette.h"

#include "cassedoc.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

//Version number for cassette documents. 
//This way, future versions can read/convert old versions.
static const LONG DOC_VERSION  = 100;
//This application uses MM_LOENGLISH, which is 100 pixels
// per inch.
static const short PIXPERINCH = 100; 

//There are 72 points per inch.
static const short POINTSPERINCH = 72;

//Little constant to make debug output easier:
#ifdef _DEBUG
static const CString eol ="\n";
#endif

/////////////////////////////////////////////////////////////////////////////
// CCassetteDoc

IMPLEMENT_DYNCREATE(CCassetteDoc, CDocument)

BEGIN_MESSAGE_MAP(CCassetteDoc, CDocument)
	//{{AFX_MSG_MAP(CCassetteDoc)
	ON_UPDATE_COMMAND_UI(ID_FILE_SAVE, OnUpdateFileSave)
	ON_COMMAND(ID_FONTS_ALBUM, OnFontsAlbum)
	ON_COMMAND(ID_FONTS_ARTIST, OnFontsArtist)
	ON_COMMAND(ID_FONTS_NOTES, OnFontsNotes)
	ON_COMMAND(ID_FONTS_SONGS, OnFontsSongs)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CCassetteDoc construction/destruction

//Initialize all the members to known states.  
//The CStrings are initialized using the default constructors.
//InitFont initializes all the fonts.
//
//PreConditions:  None.
//PostConditions:  A valid object.
CCassetteDoc::CCassetteDoc() :
	m_lVersion (DOC_VERSION)
{
InitFont (&m_lfontSongs);
InitFont (&m_lfontArtist);
InitFont (&m_lfontAlbum);
InitFont (&m_lfontNotes);

//PostConditions:
ASSERT_VALID (this);
}

//Destructor:  Does nothing.
CCassetteDoc::~CCassetteDoc()
{
}

//This initializes the logical font structure passed in to a 
//MS Sans Serif 8 point font.  The other constants used 
//define the mapping mode and the size in inches.
//
//PreConditions: Font is non-null.
//PostConditions: A valid logical font. (not enforced).
void CCassetteDoc::InitFont (LOGFONT* Font)
{
//PreConditions:
//This is called during construction, so don't assume a 
//valid object.
ASSERT (NULL != Font);

Font->lfHeight = (-(PIXPERINCH * 8 / POINTSPERINCH));
Font->lfWidth  = 0;
Font->lfEscapement  = 0;
Font->lfOrientation = 0;
Font->lfWeight = 0;
Font->lfItalic = 0;
Font->lfUnderline = 0;
Font->lfStrikeOut = 0;
Font->lfCharSet    = ANSI_CHARSET;
Font->lfOutPrecision = OUT_DEFAULT_PRECIS;
Font->lfClipPrecision = CLIP_DEFAULT_PRECIS;
Font->lfQuality = DEFAULT_QUALITY;
Font->lfPitchAndFamily = DEFAULT_PITCH | FF_DONTCARE;
strncpy(Font->lfFaceName, "MS Sans Serif", LF_FACESIZE);
//PostConditions:
//Not enforced.
}

//Clear this document for reuse.  
//This sets all the string values to NULL strings.
//The version is set to the current version.  The 
//fonts are not reset.  They are preserved from tape to tape.
//PreConditions: A valid object.
//PostConditions: A valid empty object (not enforced)
void CCassetteDoc::DeleteContents ()
{
//PreConditions:
ASSERT_VALID (this);

m_lVersion = DOC_VERSION;
m_csAlbum1.Empty ();
m_csAlbum2.Empty ();
m_csArtist1.Empty ();
m_csArtist2.Empty ();
m_csSongs1.Empty ();
m_csSongs2.Empty ();
m_csNotes.Empty ();
//PostConditions:
ASSERT_VALID (this);
}

/////////////////////////////////////////////////////////////////////////////
// CCassetteDoc serialization

//This utility function reads the information in the 
// font parameter from the archive object specified by ar.
//The casts to longs are to get the default serialization. 
// ints do not have serialize functions, because their size may change.
//
//Preconditions: A valid object, the archive is loading, Font is non-Null.
//PostConditions: Read everything.  (Enforced by checking the validity of this.
void CCassetteDoc::ReadFont (CArchive& ar, LOGFONT* font)
{
ASSERT_VALID (this);
ASSERT (ar.IsLoading ());
ASSERT (NULL != font);

CString FaceName;
ar >> (LONG&)font->lfHeight;
ar >> (LONG&)font->lfWidth;
ar >> (LONG&)font->lfEscapement;
ar >> (LONG&)font->lfOrientation;
ar >> (LONG&)font->lfWeight;
ar >> font->lfItalic;
ar >> font->lfUnderline;
ar >> font->lfStrikeOut;
ar >> font->lfCharSet;
ar >> font->lfOutPrecision;
ar >> font->lfClipPrecision;
ar >> font->lfQuality;
ar >> font->lfPitchAndFamily;
ar >> FaceName;
strncpy (font->lfFaceName, FaceName, LF_FACESIZE);

ASSERT_VALID (this);
}

//This utility function writes the information in the 
// font parameter to the archive object specified by ar.
//The casts to longs are to get the default serialization. 
// ints do not have serialize functions, because their size may change.
//
//PreConditions:  Valid object; archive is storing, font is non-null.
//PostConditions: Still a valid object.
void CCassetteDoc::WriteFont (CArchive& ar, LOGFONT* font)
{
//PreConditions:
ASSERT_VALID (this);
ASSERT (ar.IsStoring ());
ASSERT (NULL != font);

ar << (LONG) font->lfHeight;
ar << (LONG) font->lfWidth;
ar << (LONG) font->lfEscapement;
ar << (LONG) font->lfOrientation;
ar << (LONG) font->lfWeight;
ar << font->lfItalic;
ar << font->lfUnderline;
ar << font->lfStrikeOut;
ar << font->lfCharSet;
ar << font->lfOutPrecision;
ar << font->lfClipPrecision;
ar << font->lfQuality;
ar << font->lfPitchAndFamily;
ar << font->lfFaceName;

//PostConditions:
ASSERT_VALID (this);
}

//This private function reads a document from an archive 
// object.  It serializes the simple types and then 
// calls other private utility functions to finish
// the job.
//
//PreConditions: a valid object, as is Loading.
//PostConditions: a valid object.
void CCassetteDoc::ReadDocument (CArchive& ar)
{
//PreConditions:
ASSERT_VALID (this);
ASSERT (ar.IsLoading ());

ar >> m_lVersion;
ar >> m_csAlbum1;
ar >> m_csAlbum2;
ar >> m_csArtist1;
ar >> m_csArtist2;
ar >> m_csSongs1;
ar >> m_csSongs2;
ar >> m_csNotes;
ReadFont (ar, &m_lfontSongs);
ReadFont (ar, &m_lfontArtist);
ReadFont (ar, &m_lfontAlbum);
ReadFont (ar, &m_lfontNotes);

//PostConditions:
ASSERT_VALID (this);
}

//This private function writes a document to the archive file.
// It seraializes simple objects and calls other functions to
// write the more complex font information.
//
//PreConditions:  a valid object, archive is storing.
//PostConditions: a valid object.
void CCassetteDoc::WriteDocument (CArchive& ar)
{
//PreConditions:
ASSERT_VALID (this);
ASSERT (ar.IsStoring ());

m_lVersion = DOC_VERSION;
ar << m_lVersion;
ar << m_csAlbum1;
ar << m_csAlbum2;
ar << m_csArtist1;
ar << m_csArtist2;
ar << m_csSongs1;
ar << m_csSongs2;
ar << m_csNotes;
WriteFont (ar, &m_lfontSongs);
WriteFont (ar, &m_lfontArtist);
WriteFont (ar, &m_lfontAlbum);
WriteFont (ar, &m_lfontNotes);

//PostConditions:
ASSERT_VALID (this);
}

//This is the serialize function for the cassette label
//document.
//
//PreConditions: this is valid, ar is valid.
//PostConditions: same.
void CCassetteDoc::Serialize(CArchive& ar)
{
//PreConditions:
ASSERT_VALID (this);

if (ar.IsStoring())
	WriteDocument (ar);
else
	ReadDocument (ar);

ASSERT_VALID (this);
}

//This function changes a single font.
//The work is done by the CFontDialog.
//
//PreConditions: A valid object, valid font.
//PostConditions: still a valid object.  The return value
// is one of the expected values.
int CCassetteDoc::ChangeFont (LOGFONT* Font)
{
//PreConditions:
ASSERT_VALID (this);
ASSERT (NULL != Font);

int retval;
CFontDialog Fontdlg (Font,CF_SCREENFONTS | CF_INITTOLOGFONTSTRUCT);
ASSERT_VALID (&Fontdlg);

retval = Fontdlg.DoModal ();

//PostConditions:
ASSERT_VALID (this);
ASSERT (retval != 0);
ASSERT (IDOK == retval || IDCANCEL == retval);

return retval;
}


/////////////////////////////////////////////////////////////////////////////
// CCassetteDoc diagnostics

#ifdef _DEBUG
void CCassetteDoc::AssertValid() const
{
CDocument::AssertValid();

ASSERT (DOC_VERSION == m_lVersion);
//Can't ASSERT_VALID on CStrings, or LOGFONTS
// So here, we assert the length of the strings.
ASSERT (m_csAlbum1.GetLength () <= 256);
ASSERT (m_csAlbum2.GetLength () <= 256);
ASSERT (m_csArtist1.GetLength () <= 128);
ASSERT (m_csArtist2.GetLength () <= 128);
ASSERT (m_csSongs1.GetLength () <= 1024);
ASSERT (m_csSongs2.GetLength () <= 1024);
ASSERT (m_csNotes.GetLength () <= 1024);
//Skip the LogFonts.  It is not clear exactly what 
// makes a valid LOGFONT.  
}

//Dump.
// Print all the members of this object.  This
// definitely breaks my small function rule, but it
// is just a a debug member.  It makes no difference
// to the performance.
void CCassetteDoc::Dump(CDumpContext& dc) const
{
CDocument::Dump(dc);
dc << "m_lVersion: ";
dc << m_lVersion << eol;
dc << "m_csAlbum1: ";
dc << m_csAlbum1 << eol;
dc << "m_csAlbum2: ";
dc << m_csAlbum2 << eol;
dc << "m_csArtist1: ";
dc << m_csArtist1 << eol;
dc << "m_csArtist2: ";
dc << m_csArtist2 << eol;
dc << "m_csSongs1: ";
dc << m_csSongs1 << eol;
dc << "m_csSongs2: ";
dc << m_csSongs2 << eol;
dc << "m_csNotes: ";
dc << m_csNotes << eol;

dc << "m_lfontSongs: " << eol;
dc << (LONG) m_lfontSongs.lfHeight << eol;
dc << (LONG) m_lfontSongs.lfWidth << eol;
dc << (LONG) m_lfontSongs.lfEscapement << eol;
dc << (LONG) m_lfontSongs.lfOrientation << eol;
dc << (LONG) m_lfontSongs.lfWeight << eol;
dc << m_lfontSongs.lfItalic << eol;
dc << m_lfontSongs.lfUnderline << eol;
dc << m_lfontSongs.lfStrikeOut << eol;
dc << m_lfontSongs.lfCharSet << eol;
dc << m_lfontSongs.lfOutPrecision << eol;
dc << m_lfontSongs.lfClipPrecision << eol;
dc << m_lfontSongs.lfQuality << eol;
dc << m_lfontSongs.lfPitchAndFamily << eol;
dc << m_lfontSongs.lfFaceName << eol;

dc << "m_lfontArtist: " << eol;
dc << (LONG) m_lfontArtist.lfHeight << eol;
dc << (LONG) m_lfontArtist.lfWidth << eol;
dc << (LONG) m_lfontArtist.lfEscapement << eol;
dc << (LONG) m_lfontArtist.lfOrientation << eol;
dc << (LONG) m_lfontArtist.lfWeight << eol;
dc << m_lfontArtist.lfItalic << eol;
dc << m_lfontArtist.lfUnderline << eol;
dc << m_lfontArtist.lfStrikeOut << eol;
dc << m_lfontArtist.lfCharSet << eol;
dc << m_lfontArtist.lfOutPrecision << eol;
dc << m_lfontArtist.lfClipPrecision << eol;
dc << m_lfontArtist.lfQuality << eol;
dc << m_lfontArtist.lfPitchAndFamily << eol;
dc << m_lfontArtist.lfFaceName << eol;

dc << "m_lfontAlbum: ";
dc << (LONG) m_lfontAlbum.lfHeight << eol;
dc << (LONG) m_lfontAlbum.lfWidth << eol;
dc << (LONG) m_lfontAlbum.lfEscapement << eol;
dc << (LONG) m_lfontAlbum.lfOrientation << eol;
dc << (LONG) m_lfontAlbum.lfWeight << eol;
dc << m_lfontAlbum.lfItalic << eol;
dc << m_lfontAlbum.lfUnderline << eol;
dc << m_lfontAlbum.lfStrikeOut << eol;
dc << m_lfontAlbum.lfCharSet << eol;
dc << m_lfontAlbum.lfOutPrecision << eol;
dc << m_lfontAlbum.lfClipPrecision << eol;
dc << m_lfontAlbum.lfQuality << eol;
dc << m_lfontAlbum.lfPitchAndFamily << eol;
dc << m_lfontAlbum.lfFaceName << eol;

dc << "m_lfontNotes: ";
dc << (LONG) m_lfontNotes.lfHeight << eol;
dc << (LONG) m_lfontNotes.lfWidth << eol;
dc << (LONG) m_lfontNotes.lfEscapement << eol;
dc << (LONG) m_lfontNotes.lfOrientation << eol;
dc << (LONG) m_lfontNotes.lfWeight << eol;
dc << m_lfontNotes.lfItalic << eol;
dc << m_lfontNotes.lfUnderline << eol;
dc << m_lfontNotes.lfStrikeOut << eol;
dc << m_lfontNotes.lfCharSet << eol;
dc << m_lfontNotes.lfOutPrecision << eol;
dc << m_lfontNotes.lfClipPrecision << eol;
dc << m_lfontNotes.lfQuality << eol;
dc << m_lfontNotes.lfPitchAndFamily << eol;
dc << m_lfontNotes.lfFaceName << eol;
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CCassetteDoc commands

//Update the file save command.  The command is enabled if the 
// document has been modified.
//PreConditions: valid object, valid parms.
//PostCondition: same.
void CCassetteDoc::OnUpdateFileSave(CCmdUI* pCmdUI)
{
//PreConditions:
ASSERT_VALID (this);
ASSERT (ID_FILE_SAVE == pCmdUI->m_nID);

pCmdUI->Enable (IsModified ());	

//PostConditions:
ASSERT_VALID (this);
}

//Handle the fonts:album command.
// Call ChangeFont with the album font.
// Check the return to determine if the views need to be updated.
//PreConditions: valid object.
//PostConditions: same.
void CCassetteDoc::OnFontsAlbum()
{
ASSERT_VALID (this);
if (IDOK == ChangeFont (&m_lfontAlbum))
	{
	SetModifiedFlag (TRUE);
	UpdateAllViews (NULL, FONT_ALBUM_CHANGE, NULL);
	}
ASSERT_VALID (this);
}

//Handle the fonts:artist command.
// Call ChangeFont with the artist font.
// Check the return to determine if the views need to be updated.
//PreConditions: valid object.
//PostConditions: same.
void CCassetteDoc::OnFontsArtist()
{
ASSERT_VALID (this);
if (IDOK == ChangeFont (&m_lfontArtist))
	{
	SetModifiedFlag (TRUE);
	UpdateAllViews (NULL, FONT_ARTIST_CHANGE, NULL);
	}
ASSERT_VALID (this);
}

//Handle the fonts:notes command.
// Call ChangeFont with the notes font.
// Check the return to determine if the views need to be updated.
//PreConditions: valid object.
//PostConditions: same.
void CCassetteDoc::OnFontsNotes()
{
ASSERT_VALID (this);
if (IDOK == ChangeFont (&m_lfontNotes))
	{
	SetModifiedFlag (TRUE);
	UpdateAllViews (NULL, FONT_NOTES_CHANGE, NULL);
	}
ASSERT_VALID (this);
}

//Handle the fonts:songs command.
// Call ChangeFont with the songs font.
// Check the return to determine if the views need to be updated.
//PreConditions: valid object.
//PostConditions: same.
void CCassetteDoc::OnFontsSongs()
{
ASSERT_VALID (this);
if (IDOK == ChangeFont (&m_lfontSongs))
	{
	SetModifiedFlag (TRUE);
	UpdateAllViews (NULL, FONT_SONGS_CHANGE, NULL);
	}
ASSERT_VALID (this);
}

//DoFontCommand handler.
// This function is called by the view class when the user double-clicks 
// on an area of the screen.  
//PreConditions: valid object.
//PostConditions: same.
void CCassetteDoc::DoFontCommand (int WhichFont)
{
ASSERT_VALID (this);
switch (WhichFont)
	{
	case FONT_ALBUM_CHANGE:
		OnFontsAlbum ();
		break;
	case FONT_ARTIST_CHANGE:
		OnFontsArtist ();
		break;
	case FONT_SONGS_CHANGE:
		OnFontsSongs ();
		break;
	case FONT_NOTES_CHANGE:
		OnFontsNotes ();
		break;
	default:
		TRACE0 ("Danger Will Robinson, bad value passed to DoFontCommand ()\n");
		ASSERT (FALSE);
		break;
	}
ASSERT_VALID (this);
}
