/*
 *  deltree - Delete a directory tree
 *
 *  Copyright (C) 1994  Troy Rollo <troy@cbme.unsw.EDU.AU>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <stdio.h>
#include <stddef.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>
#include <stdlib.h>
#include <io.h>
#include <dir.h>
#include <dos.h>

extern	int	_rtl_chmod(const char *pchPath, int mode, ...);

static char *
expand_wildcards(char *s)
{
	static	struct find_t info;
	char wildcard[80];
	static char directory[256];
	static char filename[256];
	unsigned attrib;
	char *c1, *c2;

	if (s)
	{
		strcpy(wildcard, s);
		strcpy(directory, wildcard);
	    	if ((c1 = strrchr(directory, '/')) != 0 ||
		    (c2 = strrchr(directory, '\\')) != 0)
	    	{
			if (c1 > c2)
				c1[1] = '\0';
			else
				c2[1] = '\0';
		}
		else
			directory[0] = '\0';
		if (! strchr(wildcard, '.'))
			strcat(wildcard, "*.*");
		attrib = _A_NORMAL | _A_SUBDIR | _A_RDONLY;
		if (_dos_findfirst(wildcard, attrib, &info) != 0)
			return NULL;
		while (!strcmp(info.name, ".") || !strcmp(info.name, ".."))
			if (_dos_findnext(&info))
				return NULL;
	}
	else
	{
		if (_dos_findnext(&info) != 0)
			return NULL;
	}
	strcpy(filename, directory);
	strcat(filename, info.name);
	return filename;
}

FILE	*fpOut;

void
deltree(char	*pchFile,
	int	yes)
{
	struct find_t info;
	char wildcard[256];
	char filename[256];
	char achLine[256];
	int	result;
	struct	stat	sbuf;

	if (*pchFile == '.')
	{
		fprintf(stderr, "Cannot remove %s\n", pchFile);
		exit(1);
	}
	if (stat(pchFile, &sbuf) == -1)
	{
		fprintf(stderr, "%s: %s\n", pchFile, sys_errlist[errno]);
		exit(1);
	}
	if (!(sbuf.st_mode & S_IFDIR))
	{
		fprintf(stderr, "%s: Not a directory\n", pchFile);
		exit(1);
	}
	if (!yes && fpOut)
	{
		fprintf(fpOut, "Delete %s and all its subdirectories? (y/N) ", pchFile);
		gets(achLine);
		if (*achLine != 'Y' && *achLine != 'y')
			return;
	}
	strcpy(wildcard, pchFile);
	strcat(wildcard, "\\*.*");
	for (result = _dos_findfirst(wildcard,
				_A_NORMAL |
				_A_RDONLY |
				_A_HIDDEN |
				_A_SYSTEM |
				_A_SUBDIR |
				_A_ARCH,
				&info);
	     result == 0;
	     result = _dos_findnext(&info))
	{
		if (*info.name == '.')
			continue;
		strcpy(filename, pchFile);
		strcat(filename, "\\");
		strcat(filename, info.name);
		if (info.attrib & _A_SUBDIR)
		{
			deltree(filename, 1);
		}
		else
		{
			if (_rtl_chmod(filename, 1, 0) == -1 ||
			    unlink(filename) == -1)
			{
				fprintf(stderr, "%s: %s\n",
						filename,
						sys_errlist[errno]);
				exit(1);
			}
		}
	}
	if (_rtl_chmod(pchFile, 1, 0) == -1 ||
	    rmdir(pchFile) == -1)
	{
		fprintf(stderr, "%s: %s\n",
				pchFile,
				sys_errlist[errno]);
		exit(1);
	}
}

int
main(	int	argc,
	char	**argv)
{
	int	yes = 0;
	char	*pchFile;

	if (!isatty(0) ||
	    (!isatty(1) &&
	     !isatty(2)))
		yes = 1;

	if (isatty(1))
		fpOut = stdout;
	else if (isatty(2))
		fpOut = stderr;
	else
		fpOut = 0;

	while (--argc)
	{
		argv++;
		if (**argv == '/' ||
		    **argv == '-')
		{
			while (*++*argv)
			{
				switch(**argv)
				{
				case 'y':
				case 'Y':
					yes = 1;
					break;

				case 'i':
				case 'I':
					yes = 0;
					break;

				default:
					fprintf(stderr, "Usage: deltree [-{yi}] directory ...\n");
					return 1;
				}
			}
		}
		else
		{
			for (pchFile = expand_wildcards(*argv);
			     pchFile;
			     pchFile = expand_wildcards(0))
			{
				deltree(pchFile, yes);
			}
		}
	}
	return 0;
}

