//***************************************************************
// From the book "Win32 System Services: The Heart of Windows NT"
// by Marshall Brain
// Published by Prentice Hall
//
// Copyright 1994, by Prentice Hall.
//
// This code adds an ACE to an existing security descriptor.
//***************************************************************

// addace.cpp

#include <windows.h>
#include <iostream.h>

VOID main(VOID)
{
  BOOL ret;
  LONG err;
  SECURITY_DESCRIPTOR *sdData;
  SECURITY_DESCRIPTOR absSD;
  PSID psid;
  PACL pacl;
  PACL pNewACL;
  DWORD newACLSize;
  BOOL byDef;
  BOOL haveDACL;
  DWORD sizeRqd;
  SID_NAME_USE sidType;
  DWORD sidSize;
  UINT x;
  ACL_SIZE_INFORMATION aclSize;
  ACCESS_ALLOWED_ACE *pace;
  CHAR str[80];
  DWORD strSize;

  cout << "Adding \"Administrator\" "
    << "to HKEY_CURRENT_USER..."
    << endl;
  
  // ----- Get new group's SID -----

  // lookup SID from name
  sidSize=0;
  strSize=80;
  ret=LookupAccountName(NULL, "Administrator", NULL,
    &sidSize, str, &strSize, &sidType);
 
  if (sidSize)
  {
    // allocate memory for SID
    psid=(PSID) GlobalAlloc(GPTR, sidSize);
    // actually get SID
    strSize=80;
    ret=LookupAccountName(NULL, "Administrator",
      psid, &sidSize, str, &strSize, &sidType);
    if (!ret)
    {
      cerr << "Unable to get Administrator SID."
        << endl;
      return;
    }
  }
  else
  {
    cerr << "Administrator SID not available. "
      << GetLastError()

      << endl;
    return;
  }

  // ----- Get a copy of the SD/DACL -----

  // find out how much mem is needed
  // to hold existing SD w/DACL
  sizeRqd=0;
  err=RegGetKeySecurity(HKEY_CURRENT_USER,
    DACL_SECURITY_INFORMATION,
    NULL, &sizeRqd);
  if (err != ERROR_INSUFFICIENT_BUFFER)
  {
    cerr << "Unable to get SD size."
      << endl;
    return;
  }

  // allocate that memory
  sdData=(SECURITY_DESCRIPTOR *)
    GlobalAlloc(GPTR, sizeRqd);
  if (sdData == NULL)
  {
    cerr << "Unable to allocate memory." << endl;
    return;
  }

  // actually get the SD info
  err=RegGetKeySecurity(HKEY_CURRENT_USER,
    DACL_SECURITY_INFORMATION,
    sdData, &sizeRqd);
  if (err != ERROR_SUCCESS)
  {
    cerr << "Unable to get SD info." << endl;
    return;
  }

  // ----- Create a new absolute SD and DACL -----

  // initialize absolute SD
  ret=InitializeSecurityDescriptor(&absSD,
    SECURITY_DESCRIPTOR_REVISION);
  if (!ret)
  {
    cerr << "Unable to init new SD."
      << endl;
    return;
  }

  // get the DACL info
  ret=GetSecurityDescriptorDacl(sdData,
    &haveDACL, &pacl, &byDef);
  if (!ret)
  {
    cerr << "Unable to get DACL info." << endl;
    return;
  }

  if (!haveDACL)
  {
    // compute size of new DACL
    newACLSize= sizeof(ACCESS_ALLOWED_ACE) +
      GetLengthSid(psid) - sizeof(DWORD);
  }
  else
  {
    // get size info about existing DACL
    ret=GetAclInformation(pacl, &aclSize,
      sizeof(ACL_SIZE_INFORMATION),
      AclSizeInformation);

    // compute size of new DACL
    newACLSize=aclSize.AclBytesInUse +
      sizeof(ACCESS_ALLOWED_ACE) +
      GetLengthSid(psid) -
      sizeof(DWORD);
  }

  // allocate memory
  pNewACL=(PACL) GlobalAlloc(GPTR, newACLSize);
  if (pNewACL == NULL)
  {
    cerr << "Unable to allocate memory." << endl;
    return;
  }

  // initialize the new DACL
  ret=InitializeAcl(pNewACL, newACLSize,
    ACL_REVISION);
  if (!ret)
  {
    cerr << "Unable to init new DACL."
      << endl;
    return;
  }

  // ----- Copy existing DACL into new DACL -----

  if (haveDACL)
  {
    // copy ACEs from existing DACL
    // to new DACL
    for (x=0; x<aclSize.AceCount; x++)
    {
      ret=GetAce(pacl, x, (LPVOID *) &pace);
      if (!ret)
      {
        cerr << "Unable to get ACE."
          << endl;
        return;
      }

      ret=AddAce(pNewACL, ACL_REVISION, MAXDWORD,
        pace, pace->Header.AceSize);
      if (!ret)
      {
        cerr << "Unable to add ACE."
          << endl;
        return;
      }
    }
  }

  // ----- Add the new ACE to the new DACL -----

  // add access allowed ACE to new
  // DACL
  ret=AddAccessAllowedAce(pNewACL,
    ACL_REVISION, GENERIC_ALL, psid);
  if (!ret)
  {
    cerr << "Unable to add ACE."
      << endl;
    return;
  }

  // set the new DACL
  // in the absolute SD
  ret=SetSecurityDescriptorDacl(&absSD,
    TRUE, pNewACL, FALSE);
  if (!ret)
  {
    cerr << "Unable to install DACL."
      << endl;
    return;
  }

  // check the new SD
  ret=IsValidSecurityDescriptor(&absSD);
  if (!ret)
  {
    cerr << "SD invalid."
      << endl;
    return;
  }

  // ----- Install the new DACL -----

  // install the updated SD
  err=RegSetKeySecurity(HKEY_CURRENT_USER,
    DACL_SECURITY_INFORMATION,
    &absSD);
  if (err != ERROR_SUCCESS)
  {
    cerr << "Unable to set registry key SD."
      << endl;
  }

  // ensure that SD has been updated
  RegCloseKey(HKEY_CURRENT_USER);

  // release memory
  if(GlobalFree(pNewACL))
  {
    cerr << "Unable to free memory." << endl;
    return;
  }

  // release memory
  if(GlobalFree(psid))
  {
    cerr << "Unable to free memory." << endl;
    return;
  }

  // release memory
  if(GlobalFree(sdData))
  {
    cerr << "Unable to free memory." << endl;
    return;
  }

  cout << "Done." << endl;
}
