/*  Project filter
					DCS, UP Olomouc
					Copyright  1995. All Rights Reserved.

					SUBSYSTEM:    filter.exe Application
					FILE:         fourier.cpp
					AUTHOR:       Robert Batusek

		implementace tridy TFourierFilter - tridy realizujici Fourierovu transformaci

		implementation of TFourierFilter class - class realizing Fourier transformation
*/
#pragma hdrstop

#include <math.h>

#include "fourier.h"


/////////////////////////////////////////////////
// class TComplexFilter
// --------------------
// Popis: trida obsahujici komplexni matici
// Description: class containing Complex matrix
//
TComplexFilter::TComplexFilter(TMatrix &m, BOOL CreateCopy)
	:TFilter(m,FALSE)
{
	matrix = new TComplexMatrix(m);
	if (!CreateCopy)
		delete (&m);
}

TComplexFilter::TComplexFilter(TComplexMatrix &m, BOOL CreateCopy)
	:TFilter(m,FALSE)
{
	matrix = new TComplexMatrix(m);
	if (!CreateCopy)
		delete (&m);
}

BOOL TComplexFilter::SetMatrix(TMatrix &m,BOOL CreateCopy)
{
	if (matrix) delete matrix;
	matrix = new TComplexMatrix(m);
	if (!CreateCopy)
		delete (&m);
	return TRUE;
}

BOOL TComplexFilter::FinishFiltering()
{
	//calculating the Fourier Spectrum and storing it into matrix
	((TComplexMatrix *)matrix)->ConvertComplexToBMP();

	return TFilter::FinishFiltering();
}


/////////////////////////////////////////////////
// class TFTFilter
// ---------------
// Popis: bazova trida pro realizaci FFT (rychla Fourierova transformace)
// Description: base class used for realizing FFT (fast Fourier transform)
//
void TFTFilter::RowTransform(long from, long to)
{int i,j;

	//executing 1D FFT for rows
	for(i=(int)from;i<(int)to;i++) {
		//filling 1D complex array by specified row
		for (j=0;j<=outN;j++) {
			p[j]=((TComplexMatrix *)matrix)->GetComplex(j,i);
			if ((j + i) & 1) p[j] = - p[j]; // centering the spectrum (see theory)
		}

		//1D FFT
		LineTransform(p,outN,0);

		//storing results mutiplied by outN (see theory)
		for (j=0;j<=outN;j++)
			((TComplexMatrix *)matrix)->SetComplex(j,i,p[j]*outN);
	}
}

void TFTFilter::ColumnTransform(long from, long to)
{int i,j;
	//executing 1D FFT for columns
	for (i=(int)from;i<(int)to;i++) {
		//filling 1D complex array by specified column
		for (j=0;j<=outN;j++)
			p[j]=((TComplexMatrix *)matrix)->GetComplex(i,j);

		//1D FFT
		LineTransform(p,outN,0);

		//storing results
		for (j=0;j<=outN;j++)
			((TComplexMatrix *)matrix)->SetComplex(i,j,p[j]);
	}
}

void TFTFilter::FilterImage(int start,int count)
{	long from,to,size=outN+1;

	if (start+count<=50) {
		from = size*start/50;
		to = size*(start + count)/50;
		RowTransform(from,to);
	} else
		if (start<=50) {
			from = size*start/50;
			to = size;
			RowTransform(from,to);
			from = 0;
			to = size*(start + count)/50 - size;
			ColumnTransform(from,to);
		} else {
			from = size*start/50 - size;
			to = size*(start + count)/50 - size;
			ColumnTransform(from,to);
		}
}

BOOL TFTFilter::StartFiltering()
{
	matrix->GetSize(outN,outN);
	//automatically filled with zeros
	p = new Complex[outN];
	t = new Complex[outN];

	// filling w array with Complex roots of unity
	w = new Complex[outN];
	outN--; // outN - 1 value is used in taken algorithm of FFT (see LineTransform method)

	return TComplexFilter::StartFiltering();
}

BOOL TFTFilter::FinishFiltering()
{
	delete[] p;
	delete[] t;
	delete[] w;
	return TComplexFilter::FinishFiltering();
}

// the algorithm is taken from the book ... (I don't know the name)
void TFTFilter::LineTransform(Complex p[], int N, int k)
{
	int i,j;
	Complex p0,p1;

	if (N==1) {
		p0 = p[k];
		p1 = p[k+1];
		p[k] = p0 + p1;
		p[k+1] = p0 - p1;
	} else {
		for (i=0,j=k; i<= N/2; i++,j++) {
//			j = k + 2*i;
			t[i] = p[j];
			t[i + 1 + N/2] = p[++j];
		}

		for (i=0;i<=N;i++)
			p[k+i] = t[i];
		LineTransform(p,N/2,k);
		LineTransform(p,N/2,k+1+N/2);

		j = (outN+1)/(N+1);
		for (i=0; i<=N/2;i++) {
			p0 = w[i*j] *p[k+N/2+1+i];
			t[i] = p[k+i] + p0;
			t[i+N/2+1] = p[k+i] - p0;
		}
		for (i=0;i<=N;i++) p[k+i] = t[i];
	}
}

/////////////////////////////////////////////////
// class TFFT
// -------------
// Popis: trida realizujici FFT (rychlou Fourierovu transformaci)
// Description: class realizing FFT (fast Fourier transform)
//
BOOL TFFT::StartFiltering()
{ int i;

	BOOL result = TFTFilter::StartFiltering();

	for (i=0;i<=outN;i++)
		w[i] = Complex(cos( (2*M_PI*i) / outN), - sin( (2*M_PI*i) / outN));
	return result;
}


/////////////////////////////////////////////////
// class TIFFT
// -------------
// Popis: trida realizujici inverzni FFT (rychlou Fourierovu transformaci)
// Description: class realizing inverse FFT (fast Fourier transform)
//
void TIFFT::RowTransform(long from, long to)
{int i,j;

	//executing 1D FFT for rows
	for(i=(int)from;i<(int)to;i++) {
		//filling 1D complex array by specified row
		for (j=0;j<=outN;j++)
			p[j]=((TComplexMatrix *)matrix)->GetComplex(j,i);

		//1D FFT
		LineTransform(p,outN,0);

		//storing results mutiplied by outN (see theory)
		for (j=0;j<=outN;j++)
			((TComplexMatrix *)matrix)->SetComplex(j,i,p[j]*outN);
	}
}


BOOL TIFFT::StartFiltering()
{ int i;

	BOOL result = TFTFilter::StartFiltering();

	for (i=0;i<outN;i++)
		w[i] = Complex(cos( (2*M_PI*i) / outN),sin( (2*M_PI*i) / outN));

	return result;
}

BOOL TIFFT::FinishFiltering()
{
	delete[] p;
	delete[] t;
	delete[] w;
	((TComplexMatrix *)matrix)->MoveComplexToBMP();
	return TFilter::FinishFiltering();
}


/////////////////////////////////////////////////
// class TFrequencyFilter
// --------------------
// Popis: abstraktni trida predka pro filtry ve frekvencni oblasti
// Description: abstract base class for filters in frequency spectrum
//
TFrequencyFilter::TFrequencyFilter(TComplexMatrix &m, BOOL CreateCopy,int r)
	:TComplexFilter(m,CreateCopy)
{
	radius = r;
  matrix->GetSize(xsize,ysize);
	xmiddle = xsize/2;
	ymiddle = ysize/2;
}

void TFrequencyFilter::FilterImage(int start,int count)
{ int j,i;
	long startrow,endrow;

	matrix->GetSize(xsize,ysize);
	startrow = (long)ysize*start/100;
	endrow = (long)ysize*(start + count)/100;
	for (i=(int)startrow;i<endrow;i++)
		for (j=0;j<xsize;j++)
			if (ChangePixel(j,i))
				((TComplexMatrix *)matrix)->SetComplex(j,i,FilterComplexElement(j,i));
}

#pragma argsused
Complex TFrequencyFilter::FilterComplexElement(int x,int y)
{
	 return Complex(0.0,0.0);
}



/////////////////////////////////////////////////
// class TLowPass
// --------------------
// Popis: aDolni propust
// Description: lowpass filtering
//
BOOL TLowPass::ChangePixel(int x, int y)
{double sqrdistance;

	sqrdistance = (x - xmiddle)*(x - xmiddle) + (y - ymiddle)*(y - ymiddle);
	return (sqrdistance>(radius*radius));
}


/////////////////////////////////////////////////
// class THighPass
// --------------------
// Popis: Horni propust
// Description: highpass filtering
//
BOOL THighPass::ChangePixel(int x, int y)
{double sqrdistance;

	sqrdistance = (x - xmiddle)*(x - xmiddle) + (y - ymiddle)*(y - ymiddle);
	return (sqrdistance<=(radius*radius));
}


/////////////////////////////////////////////////
// class TBandFilter
// --------------------
// Popis: Pasmovy filtr
// Description: Band filter
//
TBandFilter::TBandFilter(TComplexMatrix &m, BOOL CreateCopy,float r,float r2)
			:TFrequencyFilter(m,CreateCopy,r)
{
	secondradius = r2;
}


/////////////////////////////////////////////////
// class TBandPass
// --------------------
// Popis: Pasmova propust
// Description: bandpass filtering
//
BOOL TBandPass::ChangePixel(int x, int y)
{double sqrdistance;

	sqrdistance = (x - xmiddle)*(x - xmiddle) + (y - ymiddle)*(y - ymiddle);
	return ( (sqrdistance<=(radius*radius))  || (sqrdistance>(secondradius*secondradius)) );
}


/////////////////////////////////////////////////
// class TBandStop
// --------------------
// Popis: Opak pasmove propusti (nevim, jak to prelozit)
// Description: bandstop filtering
//
BOOL TBandStop::ChangePixel(int x, int y)
{double sqrdistance;

	sqrdistance = (x - xmiddle)*(x - xmiddle) + (y - ymiddle)*(y - ymiddle);
	return ( (sqrdistance>(radius*radius))  &&  (sqrdistance<=(secondradius*secondradius)) );
}


/////////////////////////////////////////////////
// class TButterworthLowpass
// --------------------
// Popis: filtry pana Butterwortha - dolni propust
// Description: Butterworth's filters - lowpass
//
Complex TButterworthLowpass::FilterComplexElement(int x,int y)
{double sqrdistance,ratio = 0;

	if ( (x!=xmiddle) || (y!=ymiddle) ) {
		sqrdistance = (x - xmiddle)*(x - xmiddle) + (y - ymiddle)*(y - ymiddle);
		ratio = 1 / (1 + (radius*radius)/sqrdistance);
	}

	return (((TComplexMatrix *)matrix)->GetComplex(x,y)) * ratio;
}


/////////////////////////////////////////////////
// class TButterworthHighpass
// --------------------------
// Popis: filtry pana Butterwortha - horni propust
// Description: Butterworth's filters - highpass
//
Complex TButterworthHighpass::FilterComplexElement(int x,int y)
{double sqrdistance,ratio = 0;

	if ( (x!=xmiddle) || (y!=ymiddle) ) {
		sqrdistance = (x - xmiddle)*(x - xmiddle) + (y - ymiddle)*(y - ymiddle);
		ratio = (1 / (1 + sqrdistance/(radius*radius)));
	}

 return ((TComplexMatrix *)matrix)->GetComplex(x,y) * ratio;
}
