/*  Project filter
		DCS, UP Olomouc
		Copyright  1995. All Rights Reserved.

		SUBSYSTEM:    filter.exe Application
		FILE:         filter.h
		AUTHOR:       Robert Batusek

		deklarace tridy TFilter - rodicovske tridy vsech filtrovacich trid
		deklarace nekterych trid pro filtracni metody
		deklarace nekterych pomocnych trid ppotrebnych k filtraci

		declaration of TFilter class - parent of all filter classes
		declaration of some filter methods
		declaration of help classes needed to filtering
*/

#ifndef _filter_h
#define _filter_h

#include <stdlib.h>
#include "matrix.h"


/////////////////////////////////////////////////
// class TFilter
// -------------
// Popis: rodicovska trida vsech filtrovacich trid
// Description: base class of all filter methods
//
class TFilter {
	public:
		TFilter(TMatrix &m,BOOL CreateCopy);
		virtual ~TFilter();

		virtual BOOL SetMatrix(TMatrix &m,BOOL CreateCopy);
		TMatrix *GetMatrix();
		virtual void FilterImage(int start = 0,int count = 100);
		virtual BOOL StartFiltering(){return TRUE;}
		virtual BOOL FinishFiltering(){return TRUE;}

	protected:
		int xsize,ysize;
		TMatrix *matrix;

		virtual Element FilterElement(int x,int y) = 0;
};

/////////////////////////////////////////////////
// class TDigitalize
// -------------
// Popis: testovaci trida, digitalizuje obrazek
// Descritpion: test class, digitalizes picture
//
class TDigitalize: public TFilter {
	public:
		TDigitalize(TMatrix &m,BOOL CreateCopy,int newsize = 2);

	protected:
		int size;

		virtual Element FilterElement(int x,int y);
};

/////////////////////////////////////////////////
// class TNoise
// -------------
// Popis: testovaci trida, vytvori na obrazku umele sum
// Description: test class, adds a noise to picture
//
class TNoise: public TFilter {
	public:
		TNoise(TMatrix &m,BOOL CreateCopy,BOOL newtype=FALSE, int newpercent = 3,int newsize = 5);

		virtual void FilterImage(int start = 0,int count = 100);
		virtual BOOL StartFiltering() {randomize(); return TRUE;}

	protected:
		BOOL type;
		int percent,deviation;

		virtual Element FilterElement(int x,int y);
};

/////////////////////////////////////////////////
// class TNegative
// -------------
// Popis: udela z obrazku jeho negativ
// Descritpion: makes a negativ from the picture
//
class TNegative: public TFilter {
	public:
		TNegative(TMatrix &m,BOOL CreateCopy):TFilter(m,CreateCopy){};

	protected:
		virtual Element FilterElement(int x,int y);
};

/////////////////////////////////////////////////////////////////////////
// class TEnvironsFilter
// -------------
// Popis: zakladni trida pro metody filtrace, ktere vyuzivaji k filtraci
//				okoli zpracovavneho pixelu
// Description: base class for those methods, which use environs to filtering -
//
class TEnvironsFilter: public TFilter {
	public:
		TEnvironsFilter(TMatrix &m,BOOL CreateCopy,int newxsize,int newysize = 0);
		~TEnvironsFilter();

		virtual void FilterImage(int start = 0,int count = 100);
		virtual BOOL StartFiltering();
		virtual BOOL FinishFiltering();

	protected:
		int xenvsize,yenvsize;  //sizes of pixel environs
		TMatrix *newmatrix;     //copy of picturedata used to filtering

		void GetLimits(int x,int y,int &lxlimit,int &lylimit,int &rxlimit,int &rylimit);
};

/////////////////////////////////////////////////
// class TMaskedFilter
// -------------
// Popis: zakladni trida pro metody filtrace, ktere vyuzivaji k filtraci masku
// Description: base class for those methods, which use mask to filtering
//
class TMaskedFilter: public TEnvironsFilter {
	public:
		TMaskedFilter(TMatrix &m,BOOL CreateCopy);
		~TMaskedFilter();

		virtual BOOL StartFiltering();

	protected:
		TMask *mask;

		virtual Element FilterElement(int x,int y);
};

/////////////////////////////////////////////////
// class TAverage
// -------------
// Popis: prvni skutecne filtracni trida - filtruje tzv. prumerovanim
// Description: first really filter class - filtering by average of surrounding pixels
//
class TAverage: public TMaskedFilter {
	public:
		TAverage(TMatrix &m,BOOL CreateCopy,int masksize,BOOL agauss = FALSE);

		virtual BOOL StartFiltering();

	protected:
		BOOL gauss;
};

/////////////////////////////////////////////////
// class TModifiedAverage
// -------------
// Popis: modifikovane prumerovani - odchylka od puvodniho jasu nesmi presahnout urcitou mez
// Description: modified average - deviation of origin and new brightness must be less than limit
//
class TModifiedAverage: public TAverage {
	public:
		TModifiedAverage(TMatrix &m,BOOL CreateCopy,int masksize,BOOL agauss = FALSE)
			: TAverage(m,CreateCopy,masksize,agauss){};

		virtual BOOL StartFiltering();

	protected:
		Element limit;
		virtual Element FilterElement(int x,int y);
};

/////////////////////////////////////////////////
// class TMedian
// -------------
// Popis: nova hodnota je rovna medianu pixelu v jeho okoli
// Description: new value is median of surrounding pixels
//
class TMedian: public TEnvironsFilter {
	public:
		TMedian(TMatrix &m,BOOL CreateCopy,int newsize)
			:TEnvironsFilter(m,CreateCopy,newsize){};

		virtual BOOL StartFiltering();
		virtual BOOL FinishFiltering();
	protected:
		Element *succesion;
		
		virtual Element FilterElement(int x,int y);
};

/////////////////////////////////////////////////
// class TRotatingMaskFilter
// -------------
// Popis: pomoci rotujici masky se urci ta cast okoli, ke ktere pixel patri
//				a z te se pocita prumer
// Description: rotating mask determines part of environs which pixel belongs to
//				and then the average is calculated from this part
//
class TRotatingMaskFilter: public TMaskedFilter {
	public:
		TRotatingMaskFilter(TMatrix &m,BOOL CreateCopy);

		virtual BOOL StartFiltering();
		virtual BOOL FinishFiltering();

	protected:
		Element *average;
		int *deviation;

		virtual Element FilterElement(int x,int y);
};

/////////////////////////////////////////////////
// class TDifference
// -------------
// Popis: detekto hran - odecte puvodni obraz od obrazu prumerovaneho
// Description: edge detector - use subtraction between original and averaged image
//
class TDifference: public TAverage {
	public:
		TDifference(TMatrix &m,BOOL CreateCopy,int masksize,int i)
			: TAverage(m,CreateCopy,masksize,FALSE){intensity = i;}

	protected:
		int intensity;

		virtual Element FilterElement(int x,int y);
};


/////////////////////////////////////////////////
// class TEdgeDetector
// -------------
// Popis: zakladni detektor hran
// Description: basic edge detector
//
class TEdgeDetector: public TMaskedFilter {
	public:
		TEdgeDetector(TMatrix &m,BOOL CreateCopy)
			:TMaskedFilter(m,CreateCopy){};
	protected:
		virtual Element FilterElement(int x,int y);
};

/////////////////////////////////////////////////
// class TLaplace
// -------------
// Popis: detektor hran - tzv. Laplacuv operator
// Description: edge detector - Laplace opterator
//
class TLaplace: public TEdgeDetector {
	public:
		TLaplace(TMatrix &m,BOOL CreateCopy);
};

/////////////////////////////////////////////////
// class TSobel
// -------------
// Popis: detektor hran - tzv. Sobeluv operator
// Description: edge detector - Sobel's opterator
//
class TSobel: public TEdgeDetector {
	public:
		TSobel(TMatrix &m,BOOL CreateCopy);
	protected:
		virtual Element FilterElement(int x,int y);
	private:
		void SetDirection(int i);
};

/////////////////////////////////////////////////
// class TLaplaceSharp
// ---------------------
// Popis: ostrici operator - pouziva Laplaceuv operator k ostreni
// Description: sharpening operator - use Laplace operator to sharpening the image
//
class TLaplaceSharp: public TLaplace {
	public:
		TLaplaceSharp(TMatrix &m,BOOL CreateCopy,float i)
			:TLaplace(m,CreateCopy){intensity = i;}

	protected:
		float intensity;

		virtual Element FilterElement(int x,int y);
};

/////////////////////////////////////////////////
// class TSobelSharp
// ---------------------
// Popis: ostrici operator - pouziva Sobeluv operator k ostreni
// Description: sharpening operator - use Sobel operator to sharpening the image
//
class TSobelSharp: public TSobel {
	public:
		TSobelSharp(TMatrix &m,BOOL CreateCopy,float i)
			:TSobel(m,CreateCopy){intensity = i;}

	protected:
		float intensity;

		virtual Element FilterElement(int x,int y);
};

/////////////////////////////////////////////////
// class TFunctionalAproximation
// -----------------------------
// Popis: hleda homogenni oblasti, ktere nahrazuje prumerem hodnot
// Description: looks for homogenous areas and replaces them by mean
// see [Pavlidis82], pg. 61
class TFunctionalAproximation : public TFilter {
	public:
		TFunctionalAproximation(TMatrix &m,BOOL CreateCopy,int v)
		 :TFilter(m,CreateCopy) {maxvariance = v;}

		virtual void FilterImage(int start = 0,int count = 100);

	protected:
		Element mean,variance,maxvariance;

		virtual void FilterArea(int xleft,int xareasize, int ytop, int yareasize);
		virtual Element FilterElement(int x,int y);
};
#endif
