;
; Serial I/O communications stub for the H8 console applications.
;
; This stub occupies the 192 bytes of memory from 2040-20FF, and
; provides console serial I/O functions to an application program.
; The program would normally be origined to run at 2100, and would
; simply be appended to this image to create the complete program.
;
; Serial input is interrupt driven, and provides a 64 byte receive
; FIFO buffer.
;
; Entry points:
;  2040  = Inital run address (launch)
;  2043  = Write character in (A) to serial port
;  204F  = Wait for character from serial port
;  2056  = Test for character from serial port.
;          If no char, CF=0, A=0
;          If char, CF=1, A contains character
;
; This source file is compatible with DDS ASM85: http://www.dunfield.com
;
; Dave Dunfield - May 2004
;
CDATA	EQU	$00FA		; Console Data port
CCTRL	EQU	$00FB		; Console Control/Status port
RBSIZE	EQU	64		; Receive buffer size (must be power of 2)
;
	ORG	$2040		; Normal H8 entry point
;
	JMP	INIT		; Init system & go
;
; Write data to UART
;
WRITE:	PUSH	PSW		; Save A
wr1:	IN	CCTRL		; Read console
	ANI	%00000001	; TX ready?
	JZ	wr1		; No, wait for it
	POP	PSW		; Restore A
	OUT	CDATA		; Send it
	RET
;
; Read character - wait for input
;
READ:	CALL	TEST		; Test for character
	JNC	READ		; Wait for it
	RET
;
; Test for character available in receive buffer
;
TEST:	PUSH	H		; Save HL
	LHLD	INRPTR		; Get read pointer
	MOV	A,L		; Get write
	SUB	H		; Data in buffer?
	JZ	testn		; No data (note CF=0)
	PUSH	B		; save BC
	MVI	H,0		; Zero high
	LXI	B,INDATA	; Point to data buffer
	DAD	B		; Advance to buffer
	MOV	A,M		; Get data
	LXI	B,-(INDATA-1)	; Restore & add 1
	DAD	B		; Adjust
	POP	B		; Restore B
	PUSH	PSW		; Save char
	MOV	A,L		; Get offset
	ANI	RBSIZE-1	; Mask for wrap
	STA	INRPTR		; Resave
	POP	PSW		; Restore char
	STC			; Indicate data (CF=1)
TESTN:	POP	H		; Restore H
	RET
;
; UART interrupt:
; Read data from UART and place in circular receive buffer
;
UIRQ:	PUSH	PSW		; Save AF
	PUSH	B		; Save BC
	PUSH	HL		; Save HL
	LDA	CCTRL		; Get console status
	ANI	%00000010	; Data ready?
	JZ	IRET		; No, skip it
	LHLD	INWPTR		; Get input pointer
	MVI	H,0		; Zero high
	LXI	B,INDATA	; Point to input data
	DAD	B		; Offset to table
	IN	CDATA		; Read data
	MOV	M,A		; Save it
	LXI	B,-(INDATA-1)	; Restore offset & add 1
	DAD	B		; Adjust
	MOV	A,L		; Get data back
	ANI	RBSIZE-1	; Mask for buffer wrap
	STA	INWPTR		; Resave
IRET:	POP	HL		; Restore HL
	POP	BC		; Restore BC
	POP	PSW		; Restore AH
	EI			; Allow next interrupt
	RET
;
; Circular buffer pointers and data storage
;
INRPTR	DB	0		; Circular input buffer read pointer
INWPTR	DB	0		; Circular input write pointer
INDATA	DB	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0	; 16 bytes
	DB	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0	; 32 bytes
	DB	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0	; 48 bytes
	DB	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0	; 64 bytes
	DB	0,0,0,0		; Filler to align
;
; Initialize uart vector, uart hardware and execute application
;
INIT:	DI			; No interrupts
	MVI	A,$C3		; Jump instruction
	LXI	H,UIRQ		; Address of Interrupt handler
	STA	$2025		; Set instruction in PAM-8 030 vector
	SHLD	$2026		; Set offset in PAM-8 030 vector
	MVI	A,%10000001	; Insure NOT in reset state
	OUT	CCTRL		; Write to console control
	MVI	A,%01000000	; Force internal RESET (now we are sure)
	OUT	CCTRL		; Write to console control
	MVI	A,%01001110	; 1stop, Noparity, 8data, x16
	OUT	CCTRL		; Write to console control
	MVI	A,%00010111	; RTS, Ereset, !BRK, RX, DTR, TX
	OUT	CCTRL		; Write to console control
	EI			; Allow interrupts
; Here we "fall out" of the 2040-20FF block, and into the application
; which should start at 2100
