/****************************************************/
/* Far Memory Routines for Zstrings					*/
/*													*/
/*	- provides a mechanism to store zero			*/
/*	  terminated strings (Zstring) in main memory	*/
/*	- utilizes a segment:offset data structure 		*/
/*	  (far pointer) to store and retreive strings	*/
/* 	- farputs(char *p, struct fardata *d)			*/
/*	  stores a Zstring in far data block already	*/
/*	  allocated										*/
/*	- fargets(char *p, struct fardata *d)			*/
/* 	  gets a Zstring in far data block				*/
/*	- far_puts(char *s, struct fardata *fs)			*/
/*	  allocates and stores a Zstring in far a		*/
/*	  data block									*/
/*													*/
/*		M. Watson	15 Oct 98						*/
/*													*/
/*	Micro-C/PC compile command: cc farstrng -fop	*/
/****************************************************/


/* data structure for far data */
struct fardata {
	unsigned offs;
	unsigned seg;
};

/* put Zstring in far data block already allocated */
farputs(/* char *p, struct fardata *d */) asm
{
		mov		bx,4[bp]	;dest offs
		mov		di,[bx]
		mov		es,2[bx]	;dest seg
;
		mov		si,6[bp]	;source offs
;
		xor		ah,ah		;perform Zstring move
fps1:	lodsb
		stosb
		cmp		al,ah
		jnz		fps1
}

/* get Zstring in far data block */
fargets(/* char *p, struct fardata *d */) asm
{
		push	ds
;
		mov		ax, ds		;dest seg
		mov		es, ax
		mov		di, 6[bp]	;dest offs
;	
		mov		bx, 4[bp]	;source	offs
		mov		si,	[bx]
		mov		ds, 2[bx]	;source seg
;
		xor		ah,ah		;perform Zstring move
fgs1:	lodsb
		stosb
		cmp		al,ah
		jnz		fgs1
;
		pop		ds
}

/* allocate and put Zstring in far data block */
far_puts(char *s, struct fardata *fs)
{
	int len;
	static farseg0 = 0;
	static farseg  = 0;
	static faroffs = 0;
	static newoffs = 0;
	static npages  = 0;

	/* initialize */
	if(!farseg0) {
		if(!(farseg0 = alloc_seg(0x10)))
			abort(" - not enough memory");
		farseg = farseg0;
	}

	/* calculate offset based on string length */
	len = strlen(s) + 1;

	if(faroffs % 16)				/* same page */
		newoffs += len;
	else {
		farseg += (faroffs/16);		/* new page boundary */
		faroffs = 0;
		newoffs = len;
		npages	= farseg-farseg0;
	}

	/* resize allocated memory to fit new string */
	if(resize_seg(farseg0, npages + ((newoffs+15)/16)))
		abort(" - not enough memory");

	/* store far pointer to string */
	fs->seg	 = farseg;
	fs->offs = faroffs;
	faroffs	 = newoffs;

	/* perform far memory move */
	farputs(s, fs);
}

/*
 * Test program - added by Dave Dunfield
 *
 * Store a large quantity of strings in far memory, then retreive
 * and display some of them.
 */
#include <stdio.h>
#define	STRINGS	1000		/* store 1000 strings */
main()
{
	struct fardata fs[STRINGS];
	unsigned i;
	char string[101];

	/*
	 * Store strings in far memory
	 * 77 digits + '[' + ']' + '\0' = 80 bytes/string
	 * times 1000 strings = 80k of string storage
	 */
	printf("Storing strings... ");
	for(i=0; i < STRINGS; ++i) {
		sprintf(string, "[%077u]", i);
		far_puts(string, fs[i]); }

	/*
	 * Replace an already allocated string
	 */
	sprintf(string, "*** A totally different string ***");
	farputs(string, fs[500]);

	/*
	 * Retreive strings from memory & display
	 */
	printf("Retrieving strings...\n");
	for(i=0; i < STRINGS; i += (STRINGS/20)) {
		fargets(string, fs[i]);
		printf("%s\n", string); }
}
