/*
 * An improved "pong" game...
 *
 * Much smoother ball and paddle motion.
 * Display is synchronized with vertical retrace to avoid flicker.
 * Angle of bounce can be changed by hitting ball with edge of paddle.
 * Better logic for serving the ball.
 * Ball speeds up as play progresses.
 *
 * Use LEFT-SHIFT/ALT for player 1 UP/DOWN
 * Use RIGHT-SHIFT/CTRL for player 2 UP/DOWN
 * ESCape exits the game
 *
 * Copyright 1998-2003 Dave Dunfield
 * All rights reserved.
 *
 * Permission granted for personal (non-commercial) use only.
 *
 * Compile command: cc pong1 -fop
 */
#include <lrg.h>

#define	TOP				10					/* Upper limit of playing field */
#define	BOTTOM			200					/* Lower limit of playing field */
#define	LEFT			(7+5)				/* Left  limit of playing field */
#define	RIGHT			320-(7+5)			/* Right limit of playing field */
#define	BALL_RADIUS		7					/* Radius of ball */
#define	PADDLE_WIDTH 	6					/* Width of paddle */
#define	PADDLE_HEIGHT	40					/* Height of paddle */
#define	PADDLE_EDGE		15					/* Edge zone of paddle */
#define	WIN_SCORE		10					/* Score to win match */

int
	ball_x = 320/2,							/* X position of ball */
	ball_y = 200/2,							/* Y position of ball */
	ball_xspeed,							/* X speed of ball */
	ball_yspeed,							/* Y speed of ball */
	left_player = 200/2 - PADDLE_HEIGHT/2,	/* Position of left player */
	right_player = 200/2 - PADDLE_HEIGHT/2,	/* Position of right player */
	left_score,								/* Left players score */
	right_score,							/* Right players score */
	serve;									/* Direction of next serve */

extern unsigned RAND_SEED;

main()
{
	int old_x, old_y, old_left, old_right, a, key;
	int c;

	if(lrg_open())
		abort("VGA required");

	get_time(&a, &a, &c);
	serve = (c & 1) ? 1 : -1;
	c = 900;

	/* draw scores */
	lrg_printf(  0, 0, RED<<8|B_WHITE, "Player1: %-2d", left_score);
	lrg_printf(230, 0, RED<<8|B_WHITE, "Player2: %-2d", right_score);

	for(;;) {
		lrg_fbox(0, old_left = left_player, PADDLE_WIDTH, PADDLE_HEIGHT, BLUE);
		lrg_fbox(320-PADDLE_WIDTH, old_right = right_player, PADDLE_WIDTH, PADDLE_HEIGHT, BLUE);
		lrg_fcircle(old_x = ball_x, old_y = ball_y, BALL_RADIUS, B_YELLOW);
		sound_off();

		/* Move the ball */
		ball_x += ball_xspeed;
		ball_y += ball_yspeed;
		if(ball_y < (TOP+BALL_RADIUS)) {
			ball_y = TOP+BALL_RADIUS;
			ball_yspeed = -ball_yspeed;
			sound(1300); }
		if(ball_y > (BOTTOM-BALL_RADIUS)) {
			ball_y = BOTTOM-BALL_RADIUS;
			ball_yspeed = -ball_yspeed;
			sound(1300); }
		if(ball_x < LEFT) {		/* Check for player hit */
			if(	(ball_y < (left_player-BALL_RADIUS)) ||
				(ball_y > (left_player+(BALL_RADIUS+PADDLE_HEIGHT))) ) {
				/* Player missed the ball */
				lrg_printf(230, 0, RED<<8|B_WHITE, "Player2: %-2d", ++right_score);
				beep(100, 1000);
				if(right_score >= WIN_SCORE)
					break;
				ball_x = 320/2;
				ball_y = 200/2;
				ball_xspeed = ball_yspeed = 0;
				serve = 1;
				c = 900;
				goto next; }
			sound(1100);
			ball_xspeed = -ball_xspeed;
			if(ball_y < (left_player+PADDLE_EDGE))
				--ball_yspeed;
			else if(ball_y > (left_player+(PADDLE_HEIGHT-PADDLE_EDGE)))
				++ball_yspeed; }
		else if(ball_x > RIGHT) {	/* Check for player hit */
			if(	(ball_y < (right_player-BALL_RADIUS)) ||
				(ball_y > (right_player+(BALL_RADIUS+PADDLE_HEIGHT))) ) {
				/* right player missed the ball */
				lrg_printf(  0, 0, RED<<8|B_WHITE, "Player1: %-2d", ++left_score);
				beep(100, 1000);
				if(left_score >= WIN_SCORE)
					break;
				ball_x = 320/2;
				ball_y = 200/2;
				ball_xspeed = ball_yspeed = 0;
				serve = -1;
				c = 900;
				goto next; }
			sound(1200);
			ball_xspeed = -ball_xspeed;
			if(ball_y < (right_player+PADDLE_EDGE))
				--ball_yspeed;
			else if(ball_y > (right_player+(PADDLE_HEIGHT-PADDLE_EDGE)))
				++ball_yspeed; }

		/* Place limits on ball speed */
		if(ball_yspeed < -2)
			ball_yspeed = -2;
		if(ball_yspeed > 2)
			ball_yspeed = 2;

		/* handle several key presses per update */
		for(a = 0; a < 3; ++a) {
			/* handle keyboard input */
			key = peek(0, 0x417);
			if(key & 0x0002) {	/* L-SHIFT - Player 1 up */
				if(left_player > TOP)
					--left_player; }
			if(key & 0x0008) {	/* ALT - Player 1 down */
				if(left_player < (BOTTOM-PADDLE_HEIGHT))
					++left_player; }
			if(key & 0x0001) {	/* R-SHIFT - Player 2 up */
				if(right_player > TOP)
					--right_player; }
			if(key & 0x0004) {	/* CTRL - Player 2 down */
				if(right_player < (BOTTOM-PADDLE_HEIGHT))
					++right_player; } }
		if(kbtst() == 0x1B)
			break;
		if(++c > 1000) {
			if(ball_xspeed > 0)
				++ball_xspeed;
			else if(ball_xspeed < 0)
				--ball_xspeed;
			else
				ball_xspeed = serve;
			c = 0; }
	next:
		lrg_retrace();
		lrg_fbox(320-PADDLE_WIDTH, old_right, PADDLE_WIDTH, PADDLE_HEIGHT, BLACK);
		lrg_fbox(0, old_left, PADDLE_WIDTH, PADDLE_HEIGHT, BLACK);
		lrg_fcircle(old_x, old_y, BALL_RADIUS, BLACK); }

	lrg_close();
	if(left_score > right_score)
		puts("Player 1 wins!");
	else if(left_score < right_score)
		puts("Player 2 wins!");
	else
		puts("Tie game!");
}
